package com.xebialabs.deployit.plugin.powershell;

import java.io.IOException;
import java.io.OutputStream;
import java.net.URL;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.io.Resources;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Preview;
import com.xebialabs.deployit.plugin.api.flow.PreviewStep;
import com.xebialabs.deployit.plugin.api.flow.StageableStep;
import com.xebialabs.deployit.plugin.api.flow.StagedFile;
import com.xebialabs.deployit.plugin.api.flow.StagingContext;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.rules.RulePostConstruct;
import com.xebialabs.deployit.plugin.api.rules.StepParameter;
import com.xebialabs.deployit.plugin.api.rules.StepPostConstructContext;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.deployit.plugin.powershell.PowerShellStepUtils.PowerShellScriptCallback;
import com.xebialabs.deployit.plugin.remoting.vars.VarsConverter;
import com.xebialabs.deployit.plugin.steps.CalculatedStep;
import com.xebialabs.deployit.plugin.steps.ContextHelper;
import com.xebialabs.deployit.plugin.steps.TargetContainerJavaHelper;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.plugin.powershell.PowerShellStepUtils.executePowerShellScript;
import static com.xebialabs.deployit.plugin.powershell.PowerShellStepUtils.previewPowerShellScript;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.appendScriptDir;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.appendScripts;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.loadScript;
import static com.xebialabs.overthere.OperatingSystemFamily.WINDOWS;

@SuppressWarnings("serial")
abstract class PowerShellStep extends CalculatedStep implements PreviewStep, StageableStep {

    private static final String BASE_RUNTIME_PATH = "powershell/runtime";

    @StepParameter(name = "targetHost", description = "The target host to run the script on", calculated = true)
    private HostContainer container;

    Deployed<?, ?> deployed;

    @StepParameter(name = "script", description = "Path to the Powershell script to execute (relative to XL Deploy's classpath)")
    private String scriptPath;

    @StepParameter(name = "powershellContext", description = "The variables to pass to the Powershell step", required = false, calculated = true)
    private Map<String, Object> powershellVars;

    @StepParameter(name = "uploadArtifacts", description = "If true, the artifacts from the powershell context will be uploaded to the target host and available to the script as $bindingName.file. Defaults to true.")
    private Boolean uploadArtifactData = true;

    @RulePostConstruct
    @Override
    public void doPostConstruct(StepPostConstructContext ctx) {
        super.doPostConstruct(ctx);
        if(container == null) container = TargetContainerJavaHelper.defaultTargetContainer(ctx, HostContainer.class);
        powershellVars = ContextHelper.defaultContext(ctx, powershellVars);
    }

    private List<String> classpathResources = newArrayList();

    public PowerShellStep(HostContainer container, Deployed<?, ?> deployed, String scriptPath, Map<String, Object> powerShellVars, Integer order, String description) {
        super(checkNotNull(order, "order is null"), checkNotNull(description, "description is null"));
        this.container = checkNotNull(container, "container is null");
        checkArgument(this.container.getHost().getOs() == WINDOWS, "Powershell scripts can only be run on Windows hosts");
        this.deployed = deployed;
        this.scriptPath = checkNotNull(scriptPath, "scriptPath is null");
        this.powershellVars = checkNotNull(powerShellVars, "powershellVars is null");
    }

    @SuppressWarnings("unused")
    public PowerShellStep(HostContainer container, Deployed<?, ?> deployed, String scriptPath, Map<String, Object> powerShellVars, String description) {
        this(container, deployed, scriptPath, powerShellVars, null, description);
    }

    public PowerShellStep() {
        powershellVars = Collections.emptyMap();
    }

    @Override
    public void requestStaging(StagingContext ctx) {
        if (!uploadArtifactData) {
            return;
        }

        Map<String, Object> result = new HashMap<>(powershellVars);

        for (Map.Entry<String, Object> entry : powershellVars.entrySet()) {
            if (entry.getValue() instanceof Artifact) {
                StagedFile stagedFile = ctx.stageArtifact((Artifact) entry.getValue(), container.getHost());
                result.put(entry.getKey() + VarsConverter.STAGED_FILE_VARIABLE_NAME_PREFIX, stagedFile);
            }
        }

        powershellVars = result;
    }

    @Override
    public Preview getPreview() {
        return previewPowerShellScript(getPowerShellScriptCallback(null, true));
    }

    protected StepExitCode doExecute(ExecutionContext ctx) throws Exception {
        return executePowerShellScript(container, ctx, getPowerShellScriptCallback(ctx, false));
    }

    private PowerShellScriptCallback getPowerShellScriptCallback(final ExecutionContext ctx, final boolean maskPasswords) {
        return new PowerShellStepUtils.PowerShellScriptCallback() {
            @Override
            public String getScriptPath() {
                return scriptPath;
            }

            @Override
            public String generateScript(OverthereConnection conn) {
                return doGenerateScript(conn, ctx, maskPasswords);
            }
            @Override
            public void uploadAdditionalResources(HostContainer container, ExecutionContext ctx, OverthereConnection conn) {
                doUploadClasspathResources(conn);
            }

        };
    }

    String doGenerateScript(OverthereConnection connection, ExecutionContext ctx, boolean maskPasswords) {
        logger.debug("Starting generation of PowerShell script for step [{}]", getDescription());
        StringBuilder scriptBuilder = new StringBuilder();
        logger.debug("Appending runtime scripts");
        appendBaseRuntimeScript(scriptBuilder);
        logger.debug("Appending container runtime scripts");
        appendContainerRuntimeScripts(scriptBuilder);
        logger.debug("Appending container library scripts");
        appendContainerLibraryScripts(scriptBuilder);
        logger.debug("Appending deployed library scripts");
        appendDeployedLibraryScripts(scriptBuilder);
        logger.debug("Appending PowerShell vars");
        appendPowerShellVars(connection, scriptBuilder, ctx, maskPasswords);
        logger.debug("Appending script");
        appendScript(scriptBuilder);
        logger.debug("Replacing all LF's that are NOT preceded by a CR with CRLF.");
        String script = scriptBuilder.toString().replaceAll("(?<!\r)\n", "\r\n");
        logger.debug("Finished generation of PowerShell script.");
        return script;
    }

    private void appendBaseRuntimeScript(StringBuilder b) {
        appendScriptDir(BASE_RUNTIME_PATH, b);
    }

    private void appendContainerRuntimeScripts(StringBuilder b) {
        if(container.hasProperty("runtimePath")) {
            String runtimePath = container.getProperty("runtimePath");
            appendScriptDir(runtimePath, b);
        }
    }

    private void appendContainerLibraryScripts(StringBuilder b) {
        if(container instanceof PowerShellContainer) {
            appendScripts(((PowerShellContainer) container).getLibraryScripts(), b);
        }
    }

    private void appendDeployedLibraryScripts(StringBuilder b) {
        if (deployed instanceof BasePowerShellDeployed) {
            appendScripts(((BasePowerShellDeployed<?, ?>) deployed).getLibraryScripts(), b);
        }
    }

    private void appendPowerShellVars(OverthereConnection connection, StringBuilder b, ExecutionContext ctx, boolean maskPasswords) {
        b.append("# PowerShellVars\n");
        b.append(PowerShellVarsConverter.javaToPowerShell(connection, powershellVars, uploadArtifactData, ctx, maskPasswords));
    }

    private void appendScript(StringBuilder b) {
        b.append(loadScript(scriptPath));
    }

    private void doUploadClasspathResources(OverthereConnection conn) {
        for (String r : classpathResources) {
            String filename;
            int indexOfLastSlash = r.lastIndexOf('/');
            if(indexOfLastSlash < 0) {
                filename = r;
            } else {
                filename = r.substring(indexOfLastSlash + 1);
            }

            OverthereFile target = conn.getTempFile(filename);

            logger.info("Uploading classpath resource [{}] to temporary file [{}].", r, target);
            try(OutputStream out = target.getOutputStream()) {
                URL resourceURL = Resources.getResource(r);
                Resources.copy(resourceURL, out);
            } catch (IOException exc) {
                throw new RuntimeException(exc);
            }
        }
    }

    public void setUploadArtifactData(boolean uploadArtifactData) {
        this.uploadArtifactData = uploadArtifactData;
    }

    public void setClasspathResources(List<String> classpathResources) {
        this.classpathResources = classpathResources;
    }

    private static Logger logger = LoggerFactory.getLogger(PowerShellStep.class);

}
