package com.xebialabs.deployit.plugin.powershell;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Preview;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;

import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStderr;
import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStdout;
import static com.xebialabs.deployit.plugin.remoting.preview.PreviewOverthereConnection.getPreviewConnection;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.MDC_KEY_SCRIPT_PATH;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.dumpScript;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.uploadScript;
import static com.xebialabs.overthere.util.OverthereUtils.getName;

public class PowerShellStepUtils {

    private static final String DEFAULT_POWER_SHELL_PATH = "powershell";

    static interface PowerShellScriptCallback {
        String getScriptPath();
        String generateScript(OverthereConnection conn);
        void uploadAdditionalResources(HostContainer container, ExecutionContext ctx, OverthereConnection conn);
    }

    public static Preview previewPowerShellScript(PowerShellScriptCallback callback) {
        try (OverthereConnection poc = getPreviewConnection()) {
            String scriptPath = callback.getScriptPath();
            String generatedScript = callback.generateScript(poc);
            return Preview.withSourcePathAndContents(scriptPath, generatedScript);
        }
    }

    public static StepExitCode executePowerShellScript(HostContainer container, ExecutionContext ctx, PowerShellScriptCallback callback) throws Exception {
        String scriptPath = callback.getScriptPath();
        MDC.put(MDC_KEY_SCRIPT_PATH, scriptPath);
        try (OverthereConnection conn = container.getHost().getConnection()) {
            String generatedScript = callback.generateScript(conn);
            dumpScript(getName(scriptPath), generatedScript, scriptsLogger);

            OverthereFile uploadedScriptFile = uploadScript(conn, scriptPath, generatedScript);
            callback.uploadAdditionalResources(container, ctx, conn);

            int res = executePowerShellScript(container, ctx, conn, uploadedScriptFile);

            logger.debug("Exit code: [{}]", res);
            if (res == 0) {
                return StepExitCode.SUCCESS;
            } else {
                return StepExitCode.FAIL;
            }
        } finally {
            MDC.remove(MDC_KEY_SCRIPT_PATH);
        }
    }

    private static int executePowerShellScript(HostContainer container, ExecutionContext ctx, OverthereConnection conn, OverthereFile script) {
        logger.info("Executing PowerShell script [{}] on [{}]", script, conn);
        try (DefaultExecutionOutputHandler stdoutHandler = handleStdout(ctx);
             DefaultExecutionOutputHandler stderrHandler = handleStderr(ctx)
        ) {
            CmdLine cmdLine = getScriptCommandLine(container, script);
            return conn.execute(stdoutHandler, stderrHandler, cmdLine);
        }
    }

    public static CmdLine getScriptCommandLine(HostContainer container, OverthereFile script) {
        CmdLine cmdLine;
        if (container instanceof PowerShellContainer) {
            cmdLine = ((PowerShellContainer) container).getScriptCommandLine(script);
        } else {
            cmdLine = getDefaultScriptCommandLine(DEFAULT_POWER_SHELL_PATH, script);
        }
        return cmdLine;
    }

    public static CmdLine getDefaultScriptCommandLine(String powerShellPath, OverthereFile script) {
        return CmdLine.build(powerShellPath, "-ExecutionPolicy", "Unrestricted", "-Inputformat", "None", "-NonInteractive", "-NoProfile",
            "-Command", "$ErrorActionPreference = 'Stop'; & " + script.getPath() + "; if($LastExitCode) { Exit $LastExitCode; }");
    }

    private static Logger logger = LoggerFactory.getLogger(PowerShellStepUtils.class);

    private static final Logger scriptsLogger = LoggerFactory.getLogger("com.xebialabs.deployit.plugin.powershell.scripts");

}
