package com.xebialabs.deployit.plugin.powershell;

import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.flow.*;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.deployit.plugin.powershell.PowerShellStepUtils.PowerShellScriptCallback;
import com.xebialabs.overthere.OverthereConnection;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.plugin.powershell.PowerShellStepUtils.executePowerShellScript;
import static com.xebialabs.deployit.plugin.powershell.PowerShellStepUtils.previewPowerShellScript;
import static com.xebialabs.deployit.plugin.powershell.PowerShellVarsConverter.toPowerShellString;

@SuppressWarnings("serial")
public class PowerShellBatchDeploymentStep implements Step, PreviewStep, StageableStep {

    private int order;
    private HostContainer container;
    private List<PowerShellDeploymentStep> steps;
    private String description;
    private Preview preview;

    public PowerShellBatchDeploymentStep(int order, HostContainer container, List<PowerShellDeploymentStep> steps, String description) {
        this.order = order;
        this.container = container;
        this.steps = newArrayList(steps);
        this.description = description;
    }

    @Override
    public Preview getPreview() {
        if(preview == null) {
            return previewPowerShellScript(getPowerShellScriptCallback(null));
        }
        return preview;
    }

    @Override
    public void requestStaging(StagingContext ctx) {
        for (PowerShellDeploymentStep step : steps) {
            step.requestStaging(ctx);
        }
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        return executePowerShellScript(container, ctx, getPowerShellScriptCallback(ctx));
    }

    private PowerShellScriptCallback getPowerShellScriptCallback(final ExecutionContext ctx) {
        return new PowerShellStepUtils.PowerShellScriptCallback() {
            @Override
            public String getScriptPath() {
                return "combined.ps1";
            }

            @Override
            public String generateScript(OverthereConnection conn) {
                return doGenerateScript(conn, ctx);
            }

            @Override
            public void uploadAdditionalResources(HostContainer container, ExecutionContext ctx, OverthereConnection conn) {
                //
            }
        };
    }

    public String doGenerateScript(OverthereConnection conn, ExecutionContext ctx) {
        logger.debug("Generating PowerShell batch script for step [{}]", getDescription());
        StringBuilder combinedScriptBuffer = new StringBuilder();
        for (PowerShellDeploymentStep s : steps) {
            logger.debug("Appending contents of step [{}]", s.getDescription());
            combinedScriptBuffer.append("Write-Host '===', "  + toPowerShellString(s.getDescription()) + ", '==='\r\n");
            combinedScriptBuffer.append("& {\r\n");
            combinedScriptBuffer.append(s.doGenerateScript(conn, ctx));
            combinedScriptBuffer.append("}\r\n");
        }
        return combinedScriptBuffer.toString();
    }

    @Override
    public int getOrder() {
        return order;
    }

    @Override
    public String getDescription() {
        return description;
    }

    private Logger logger = LoggerFactory.getLogger(PowerShellBatchDeploymentStep.class);
}
