package com.xebialabs.deployit.plugin.powershell;

import static com.google.common.collect.Lists.newArrayList;
import static org.apache.commons.codec.binary.Base64.encodeBase64String;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Collections2;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.remoting.vars.VarsConverter;
import com.xebialabs.overthere.OverthereConnection;

public class PowerShellVarsConverter extends VarsConverter {

    public static String javaToPowerShell(OverthereConnection connection, Map<String, Object> pythonVars) {
        return javaToPowerShell(connection, pythonVars, true);
    }

    public static String javaToPowerShell(OverthereConnection connection, Map<String, Object> pythonVars, boolean uploadArtifactData) {
        List<String> powerShellList = javaToPowerShellList(connection, pythonVars, uploadArtifactData);
        return Joiner.on("\n").join(powerShellList) + "\n";
    }

    static List<String> javaToPowerShellList(OverthereConnection connection, Map<String, Object> pythonVars, boolean uploadArtifactData) {
        PowerShellVarsConverter converter = new PowerShellVarsConverter(connection, pythonVars);
        converter.setUploadArtifactData(uploadArtifactData);
        return converter.convert();
    }

    private PowerShellVarsConverter(OverthereConnection connection, Map<String, Object> pythonVars) {
        super(connection, pythonVars);
    }

    protected void setNullVariable(String variableName) {
        add("$" + variableName + " = $null");
    }

    protected void setBooleanVariable(String variableName, boolean variableValue) {
        add("$" + variableName + " = " + toPowerShellBoolean(variableValue));
    }

    protected void setIntegerVariable(String variableName, int variableValue) {
        add("$" + variableName + " = " + variableValue);
    }

    protected void setLongVariable(String variableName, long variableValue) {
        add("$" + variableName + " = " + variableValue);
    }

    protected void setStringVariable(String variableName, String variableValue) {
        add("$" + variableName + " = " + toPowerShellString(variableValue));
    }

    private void setEmptyCollectionVar(String variableName) {
        add("$" + variableName + " = @()");
    }

    protected void setCollectionOfStringsVariable(String variableName, Collection<?> variableValue) {
        if (variableValue.isEmpty()) {
            setEmptyCollectionVar(variableName);
        } else {
            add("$" + variableName + " = " + toPowerShellCollectionOfStrings(variableValue));
        }
    }

    protected void createObject(String variableName) {
        add("$" + variableName + " = New-Object Object");
    }

    private void setNoteProperty(String objectName, String propertyName, String literalPropertyValue) {
        add("$" + objectName + " | Add-Member NoteProperty " + toPowerShellString(propertyName) + " " + literalPropertyValue);
        
    }

    protected void setNullProperty(String objectName, String propertyName) {
        setNoteProperty(objectName, propertyName, "$null");
    }

    protected void setEmptyCollectionProperty(String objectName, String propertyName) {
        setNoteProperty(objectName, propertyName, "@()");
    }

    protected void setEmptyMapProperty(String objectName, String propertyName) {
        setNoteProperty(objectName, propertyName, "@{}");
    }

    protected void setBooleanProperty(String objectName, String propertyName, boolean propertyValue) {
        setNoteProperty(objectName, propertyName, toPowerShellBoolean(propertyValue));
    }

    protected void setIntegerProperty(String objectName, String propertyName, int propertyValue) {
        setNoteProperty(objectName, propertyName, Integer.toString(propertyValue));
    }

    protected void setStringProperty(String objectName, String propertyName, String propertyValue) {
        setNoteProperty(objectName, propertyName, toPowerShellString(propertyValue));
    }

    protected void setPasswordProperty(String objectName, String propertyName, String propertyValue) {
        setNoteProperty(objectName, propertyName, "([System.Text.Encoding]::UTF8.GetString([System.Convert]::FromBase64String(" + toPowerShellString(encodeBase64String(propertyValue.getBytes())) + ")))");
    }

    protected void setCollectionOfStringsProperty(String objectName, String propertyName, Collection<?> propertyValue) {
        setNoteProperty(objectName, propertyName, toPowerShellCollectionOfStrings(propertyValue));
    }

    protected void setCiReferenceProperty(String objectName, String propertyName, ConfigurationItem propertyValue) {
        setNoteProperty(objectName, propertyName, "$" + getConfigurationItemVariableName(propertyValue));
    }

    protected void setCollectionOfCiReferencesProperty(String objectName, String propertyName, Collection<ConfigurationItem> propertyValue) {
        if (propertyValue.isEmpty()) {
            setEmptyCollectionVar(objectName + "." + propertyName);
        } else {
            List<String> varRefs = newArrayList();
            for (ConfigurationItem setItem : propertyValue) {
                varRefs.add("$" + getConfigurationItemVariableName(setItem));
            }
            setNoteProperty(objectName, propertyName, "@(" + Joiner.on(", ").join(varRefs) + ")");
        }
    }

    protected void setMapOfStringToStringReferencesProperty(String objectName, String propertyName, Map<String, String> propertyValue) {
        if (propertyValue.isEmpty()) {
            setEmptyMapProperty(objectName, propertyName);
        } else {
            List<String> entries = newArrayList();
            for (String key : propertyValue.keySet()) {
                String value = propertyValue.get(key);
                if (value == null) {
                    entries.add(toPowerShellString(key) + " = $null");
                } else {
                    entries.add(toPowerShellString(key) + " = " + toPowerShellString(value));
                }
            }
            setNoteProperty(objectName, propertyName, "@{" + Joiner.on("; ").join(entries) + "}");
        }
    }

    @Override
    protected void createObjectAndSetObjectProperty(String objectName, String propertyName) {
        String embeddedObjectVariableName = generateUniqueVariableName();
        createObject(embeddedObjectVariableName);
        setNoteProperty(objectName, propertyName, "$" + embeddedObjectVariableName);
    }

    public static String toPowerShellBoolean(Boolean bool) {
        return bool ? "$true" : "$false";
    }

    public static String toPowerShellString(String str) {
        StringBuilder converted = new StringBuilder();
        converted.append("\"");
        for (int i = 0; i < str.length(); i++) {
            char c = str.charAt(i);
            switch (c) {
            case '\"':
                converted.append('\"');
                // deliberate fall-through
            default:
                converted.append(c);
            }
        }
        converted.append("\"");
        return converted.toString();
    }

    public static String toPowerShellCollectionOfStrings(Collection<?> variableValue) {
        Collection<String> encodedStrings = Collections2.transform(variableValue, new Function<Object, String>() {
            public String apply(Object input) {
                return toPowerShellString(input.toString());
            }
        });
        return "@(" + Joiner.on(", ").join(encodedStrings) + ")";
    }

}
