import java.lang.Exception as JavaException
from com.xebialabs.deployit.engine.api.execution import TaskExecutionState as State

from policy.modules.util import minus_days_from_now


class TaskArchival:

    def __init__(self, context):
        self.logger = context.logger
        self.task_block_service = context.taskBlockService

    def run(self, archive_policy):
        self.logger.info("Running task archive job %s" % archive_policy.id)
        active_tasks = self.task_block_service.getAllCurrentTasks()
        states = TaskArchival.states_of_tasks(archive_policy)
        filtered_tasks = TaskArchival.with_statuses(active_tasks, states)
        filtered_tasks = TaskArchival.retain_days(filtered_tasks, archive_policy.taskRetention)
        self.archive_all_tasks(filtered_tasks, archive_policy.dryRun)

    @staticmethod
    def states_of_tasks(archive_policy):
        statuses = []
        if archive_policy.includeExecutedTasks:
            statuses.append(State.EXECUTED)
        if archive_policy.includeFailedTasks:
            statuses.append(State.FAILED)
        return statuses

    @staticmethod
    def with_statuses(tasks, states):
        filtered_tasks = filter(lambda task: task.state in states, tasks)
        return filtered_tasks

    @staticmethod
    def retain_days(tasks, days):
        start_date = minus_days_from_now(days)
        filtered_tasks = filter(lambda task: start_date.isAfter(task.startDate), tasks)
        return filtered_tasks

    def archive_all_tasks(self, tasks, dry_run):
        if len(tasks) is 0:
            self.logger.info("No tasks found to be archived")
        else:
            for task in tasks:
                self.archive_task(task, dry_run)
            else:
                self.logger.info("Archival is done")

    def archive_task(self, task, dry_run):
        if task.state == State.EXECUTED:
            self.execute_with_dry_run(dry_run,
                                      "Task %s will be archived now" % task.id,
                                      lambda: self.task_block_service.archive(task.id))
        elif task.state == State.FAILED:
            self.execute_with_dry_run(dry_run,
                                      "Task %s will be canceled now" % task.id,
                                      lambda: self.task_block_service.cancel(task.id))
        else:
            self.logger.error("Task %s with status %s is not supported in archival policy" % (task.id, str(task.state)))

    def execute_with_dry_run(self, dry_run, message, operation):
        if dry_run:
            self.logger.info("Dry run: " + message)
        else:
            try:
                self.logger.info(message)
                operation()
            except JavaException as exp:
                self.logger.error("Failed to archive task: " + exp.getMessage())


