#
from string import Template

from org.joda.time import DateTime
from org.joda.time.format import DateTimeFormat

from policy.modules.util import Struct
from policy.modules.tasks.export import TaskExporter
from policy.modules.tasks.collector import DefaultTaskCollector

class TaskSummary():
    def __init__(self, context):
        self.logger = context.logger

    def pretty_print(self, tasks):
        from com.xebialabs.deployit.task import TaskMetadata

        summary = {}
        for task in tasks:
            if task.metadata and TaskMetadata.APPLICATION in task.metadata:
                key = task.metadata.get(TaskMetadata.APPLICATION, "")
            elif task.metadata and TaskMetadata.TASK_TYPE in task.metadata:
                key = task.metadata.get(TaskMetadata.TASK_TYPE, "")
            else:
                key = "UNKNOWN"
            summary[key] = summary.get(key, 0) + 1

        for key, count in summary.iteritems():
            self.logger.info("- %s : %s" % (key, count))


class TaskCleaner():
    def __init__(self, context):
        self.context = Struct(context)
        self.repository_service = self.context.repositoryService
        self.task_block_service = self.context.taskBlockService
        self.logger = self.context.logger

    def _replace_placeholders(self, job):
        exec_datetime = DateTime()
        exec_date = DateTimeFormat.forPattern("yyyy-MM-dd").print(exec_datetime)
        exec_time = DateTimeFormat.forPattern("HH-mm-ss").print(exec_datetime)
        args = dict((pd.getName(), pd.get(job)) for pd in job.getType().getDescriptor().getPropertyDescriptors())
        args["execDate"] = exec_date
        args["execTime"] = exec_time
        job.archivePath = Template(job.archivePath).substitute(args)

    def _find_tasks(self, job):
        self.logger.info("Searching repository for archived tasks...")
        tasks_to_purge = DefaultTaskCollector(self.context, job.taskRetention, None).get_tasks_to_purge()
        self.logger.info("Total tasks to remove: %s" % len(tasks_to_purge))
        TaskSummary(self.context).pretty_print(tasks_to_purge)
        return tasks_to_purge

    def _export_tasks(self, job, tasks_to_purge):
        self.logger.info("Exporting tasks to archive [%s]" % job.archivePath)
        TaskExporter().export(tasks_to_purge, job.archivePath)

    def _remove_tasks(self, job, tasks_to_purge):
        self.logger.info("Removing tasks from repository...")
        for task in tasks_to_purge:
            try:
                self.logger.info("Removing task [%s]" % str(task.id))
                self.task_block_service.purge(task.id)
            except:
                self.logger.error("Error removing task [%s] from repository" % task.id)
                raise

    def run(self, job):
        self.logger.info("=== Running task purge job [%s] (taskRetention: %s, dryRun: %s) ===" % (
            job.name, job.taskRetention, job.dryRun))
        self._replace_placeholders(job)
        tasks_to_purge = self._find_tasks(job)
        if tasks_to_purge:
            if job.exportTasks:
                self._export_tasks(job, tasks_to_purge)
            if not job.dryRun:
                self._remove_tasks(job, tasks_to_purge)
        self.logger.info("=== Finished task purge job [%s] ===" % job.name)