/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.osb.ci;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;

import com.google.common.collect.Maps;
import com.xebialabs.deployit.ConfigurationItem;
import com.xebialabs.deployit.ConfigurationItemProperty;
import com.xebialabs.deployit.ConfigurationItemProperty.Size;
import com.xebialabs.deployit.ConfigurationItemRoot;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.ci.OperatingSystemFamily;
import com.xebialabs.deployit.hostsession.CmdLine;
import com.xebialabs.deployit.hostsession.CmdLineArgument;
import com.xebialabs.deployit.hostsession.CommandExecutionCallbackHandler;
import com.xebialabs.deployit.hostsession.HostFile;
import com.xebialabs.deployit.hostsession.HostFileUtils;
import com.xebialabs.deployit.hostsession.HostSession;
import com.xebialabs.deployit.plugin.wls.ci.WlsDomain;
import com.xebialabs.deployit.util.ExtendedResourceUtils;
import com.xebialabs.deployit.util.TemplateResolver;

@SuppressWarnings("serial")
@ConfigurationItem(description = "Oracle Service Bus Domain", category = "middleware", root = ConfigurationItemRoot.INFRASTRUCTURE)
public class OsbDomain extends WlsDomain {
	protected static final String OSB_WLST_PACKAGE = "com/xebialabs/deployit/plugin/osb";

	@ConfigurationItemProperty(required = true, description = "The location of the Oracle Service Bus.", size = Size.LARGE)
	private String osbHome;
	
	@ConfigurationItemProperty(description = "The version of OSB. Defaualts to OSB 3.0")
	private OsbVersion version = OsbVersion.OSB_3_0;

	public String getOsbHome() {
		return osbHome;
	}

	public void setOsbHome(String osbHome) {
		this.osbHome = osbHome;
	}

	public OsbVersion getVersion() {
		return version;
	}

	public void setVersion(OsbVersion version) {
		this.version = version;
	}

	@Override
	public int executeWlstJythonScript(CommandExecutionCallbackHandler handler, HostSession s, String scriptPath, CmdLine cmdLine) {
		List<String> cmd = new ArrayList<String>();

		switch (getActiveHost().getOperatingSystemFamily()) {
			case UNIX:
				logger.info("Uploading wlst.sh from Deployit distribution to add required libraries to the classpath.");
				HostFile uploadedWlstSh = s.copyToTemporaryFile(getWlstScript());
				cmd.add("/bin/sh");
				cmd.add(uploadedWlstSh.getPath());
				cmd.add(getWlHome());
				cmd.add(getOsbHome());
				break;
			case WINDOWS:
				logger.info("Uploading wlst.cmd from Deployit distribution to add required libraries to the classpath.");
				HostFile uploadedWlstCmd = s.copyToTemporaryFile(getWlstScript());
				cmd.add(uploadedWlstCmd.getPath());
				break;
		}

		cmd.add("-i");
		cmd.add(scriptPath);
		for (CmdLineArgument eachArgument : cmdLine.getArguments()) {
			cmd.add(eachArgument.getArg());
		}
		String[] cmdArray = cmd.toArray(new String[cmd.size()]);

		return s.execute(handler, cmdArray);
	}

	public Resource getWlstScript() {
		String extension;
		OperatingSystemFamily operatingSystemFamily = getActiveHost().getOperatingSystemFamily();
		switch (operatingSystemFamily) {
			case UNIX:
				extension = ".sh";
				break;
			case WINDOWS:
				extension = ".cmd";
				break;
			default:
				throw new IllegalArgumentException(operatingSystemFamily + " is not supported for getting the wlst script");
		}
		
		String scriptName = "wlst" + extension;

		final String template = ExtendedResourceUtils.toString(new ClassPathResource(OSB_WLST_PACKAGE + "/" + scriptName));
		Map<String, String> context = Maps.newHashMap();
		context.put("configFwkJarFile", ResourceBundle.getBundle(OSB_WLST_PACKAGE+".configfwk").getString(version.toString()));
		context.put("wlHome", getWlHome());
		context.put("osbHome", getOsbHome());
		TemplateResolver r = new TemplateResolver(context);
		final String resolvedTemplate = r.resolveLenient(template);
		
		HostSession localHostSession = Host.getLocalHost().getHostSession();
		HostFile localScriptFile = localHostSession.getTempFile("wlst", extension);
		HostFileUtils.putStringToHostFile(resolvedTemplate, localScriptFile);
		return new FileSystemResource(localScriptFile.getPath());
	}

    private static final Logger logger = LoggerFactory.getLogger(OsbDomain.class);
}
