package com.xebialabs.deployit.plugin.netscaler;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.plugin.api.deployment.planning.DefaultOrders;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.plugin.api.validation.Range;
import com.xebialabs.deployit.plugin.generic.step.WaitStep;
import com.xebialabs.deployit.plugin.lb.ci.LoadBalancer;
import com.xebialabs.deployit.plugin.netscaler.step.CliScriptExecutionStep;
import com.xebialabs.deployit.plugin.overthere.Host;

import static com.google.common.collect.FluentIterable.from;
import static com.google.common.collect.Lists.newArrayList;
import static java.lang.String.format;

@Metadata(root = Metadata.ConfigurationItemRoot.NESTED, virtual = false, description = "A Citrix NetScaler LoadBalancer")
public class NetScaler extends LoadBalancer {
    public static final String CONTAINER_NS_SHUTDOWN_DELAY = "netscalerShutdownDelay";
    static final String CONTAINER_NS_ADDRESS = "netscalerAddress";
    static final String SERVICE_GROUP = "netscalerServiceGroup";
    static final String SERVICE_GROUP_SERVICE_NAME = "netscalerServiceGroupName";
    static final String SERVICE_GROUP_SERVICE_PORT = "netscalerServiceGroupPort";


    @Property(defaultValue = "" + (DefaultOrders.STOP_ARTIFACTS - 2), hidden = true)
    private int disableServersOrder = DefaultOrders.STOP_ARTIFACTS - 2;

    @Property(defaultValue = "netscaler/disable-server.cli.ftl", hidden = true)
    private String disableServersScript = "netscaler/disable-server.cli.ftl";

    @Property(defaultValue = "" + (DefaultOrders.START_ARTIFACTS + 4), hidden = true)
    private int enableServersOrder = DefaultOrders.START_ARTIFACTS + 4;

    @Property(defaultValue = "netscaler/enable-server.cli.ftl", hidden = true)
    private String enableServersScript = "netscaler/enable-server.cli.ftl";

    @Property(defaultValue = "netscaler/run-script.cli.ftl", hidden = true)
    private String runScript = "netscaler/run-script.cli.ftl";

    @Property(required = false, description = "The amount of seconds to wait before the servers are disabled in the load balancer", defaultValue = "0")
    @Range(minimum = 0)
    private int defaultShutdownDelay;

    @Property(asContainment = true)
    private Host host;

    @Property(hidden = true, defaultValue = "source")
    private String sourceCommand;

    @Override
    public void stopTraffic(Deltas deltas, DeploymentPlanningContext ctx, Set<Container> serversToDisable) {
        Collection<NetScalerItem> nsItems;
        if (!getServiceGroups(deltas, serversToDisable).isEmpty()) {
            nsItems = newArrayList(getServiceGroups(deltas, serversToDisable));
            logger.info("Loadbalancing service groups: {}", nsItems);
        } else {
            nsItems = containersToNetScalerItems(serversToDisable);
            validateAllServersHaveNetScalerAddress(nsItems);
            logger.info("Loadbalancing servers/services : {}", nsItems);
        }
        ctx.addStep(new CliScriptExecutionStep(disableServersOrder, sourceCommand, disableServersScript, this, createContext(nsItems), format("Disable servers %s in %s", serversToDisable, this)));
        ctx.addStep(new WaitStep(disableServersOrder, determineMaxWait(nsItems), this.toString(), "disable servers"));
    }

    @Override
    public void startTraffic(Deltas deltas, DeploymentPlanningContext ctx, Set<Container> serversToEnable) {
        Collection<NetScalerItem> services;
        if (!getServiceGroups(deltas, serversToEnable).isEmpty()) {
            services = getServiceGroups(deltas, serversToEnable);

        } else {
            services = containersToNetScalerItems(serversToEnable);
            validateAllServersHaveNetScalerAddress(services);
        }
        ctx.addStep(new CliScriptExecutionStep(enableServersOrder, sourceCommand, enableServersScript, this, createContext(services), format("Enable servers %s in %s", serversToEnable, this)));
    }

    @VisibleForTesting Collection<NetScalerItem> containersToNetScalerItems(Set<Container> serversToDisable) {
        return newArrayList(Collections2.transform(serversToDisable, new Function<Container, NetScalerItem>() {
            @Override
            public NetScalerItem apply(Container input) {
                return new ServerOrService(input, defaultShutdownDelay);
            }
        }));
    }

    @VisibleForTesting List<NetScalerItem> getServiceGroups(Deltas deltas, final Set<Container> affectedServers) {
        return newArrayList(from(deltas.getDeltas()).transform(new Function<Delta, Deployed>() {
            @Override
            public Deployed apply(Delta input) {
                return input.getOperation() == Operation.DESTROY ? input.getPrevious() : input.getDeployed();
            }
        }).filter(new Predicate<Deployed>() {
            @Override
            public boolean apply(Deployed input) {
                String serviceGroup = Strings.nullToEmpty((String) input.getProperty(SERVICE_GROUP)).trim();
                return !serviceGroup.isEmpty() && affectedServers.contains(input.getContainer()) && !serviceGroup.startsWith("{{");
            }
        }).transform(new Function<Deployed, NetScalerItem>() {
            @Override
            public NetScalerItem apply(com.xebialabs.deployit.plugin.api.udm.Deployed input) {
                return new ServiceGroup(input);
            }
        }));
    }

    @VisibleForTesting int determineMaxWait(Collection<NetScalerItem> servers) {
        int waitTime = 0;
        for (NetScalerItem server : servers) {
            int serverTime = server.getDelay();
            waitTime = serverTime > waitTime ? serverTime : waitTime;
        }
        return waitTime;
    }

    private Map<String, Object> createContext(Collection<? extends NetScalerItem> servers) {
        return ImmutableMap.of("loadBalancer", this, "servers", servers);
    }

    private void validateAllServersHaveNetScalerAddress(Collection<NetScalerItem> servers) {
        for (NetScalerItem server : servers) {
            server.getAddress();
        }
    }

    @Override
    public Host getHost() {
        return host;
    }

    public void setDefaultShutdownDelay(int defaultShutdownDelay) {
        this.defaultShutdownDelay = defaultShutdownDelay;
    }

    @ControlTask(label = "Enable servers", parameterType = "netscaler.NetScalerParameters")
    public List<Step> enableService(NetScalerParameters parameters) {
        return Lists.<Step>newArrayList(new CliScriptExecutionStep(enableServersOrder, sourceCommand, enableServersScript, this, createContext(parameters.getItems()), format("Enable services %s in %s", parameters.getItems(), this)));
    }

    @ControlTask(label = "Disable servers", parameterType = "netscaler.NetScalerParameters")
    public List<Step> disableService(NetScalerParameters parameters) {
        return Lists.<Step>newArrayList(new CliScriptExecutionStep(disableServersOrder, sourceCommand, disableServersScript, this, createContext(parameters.getItems()), format("Disable services %s in %s", parameters.getItems(), this)));

    }

    @ControlTask(label = "Execute NetScaler script", parameterType = "netscaler.NetScalerScriptParameters")
    public List<Step> runScript(NetScalerScriptParameters parameters) {
        return Lists.<Step>newArrayList(new CliScriptExecutionStep(enableServersOrder, sourceCommand, runScript, this, ImmutableMap.<String, Object>of("script", parameters.getNetScalerScript()), "Run NetScaler script"));
    }

    private static final Logger logger = LoggerFactory.getLogger(NetScaler.class);
}
