package com.xebialabs.deployit.plugin.lb.planning.orchestrator;

import java.util.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.LinkedListMultimap;
import com.google.common.collect.ListMultimap;

import com.xebialabs.deployit.engine.spi.orchestration.Orchestration;
import com.xebialabs.deployit.engine.spi.orchestration.Orchestrations;
import com.xebialabs.deployit.engine.spi.orchestration.Orchestrator;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.plugin.api.udm.Environment;
import com.xebialabs.deployit.plugin.lb.ci.LoadBalancer;
import com.xebialabs.deployit.plugin.lb.planning.ContainerCollector;

import static com.google.common.collect.FluentIterable.from;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.engine.spi.orchestration.Orchestrations.interleaved;
import static com.xebialabs.deployit.plugin.lb.planning.ContainerCollector.collectContainers;
import static com.xebialabs.deployit.plugin.lb.util.Environments.getMembersOfType;
import static com.xebialabs.deployit.plugin.lb.util.LoadBalancedContainers.getAffectedWebServers;
import static com.xebialabs.deployit.plugin.lb.util.LoadBalancedContainers.getContainerToLoadBalancersMap;
import static java.lang.String.format;

@Orchestrator.Metadata(name = "sequential-by-loadbalancer-group", description = "Creates serial orchestration around load balanced servers consisting of 3 sub orchestrations: " +
        "1. disable affected servers in load balancers, 2. do actual deployment, 3. enable affected servers in load balancers")
public class LoadBalancingOrchestrator implements Orchestrator {
    private static final Type LOADBALANCER_TYPE = Type.valueOf(LoadBalancer.class);

    @Override
    public Orchestration orchestrate(DeltaSpecification specification) {
        final LoadBalancerComputedDeltas loadBalancerComputedDeltas = computeDeltas(specification);
        return Orchestrations.serial(descForAppDeployment(specification.getDeployedApplication()),
                interleaved(format("Disable %s containers in load balancers", loadBalancerComputedDeltas.getAffectedServerNameForDisable()),loadBalancerComputedDeltas.getDeltasForDisable()),
                interleaved(format("Deploy on containers %s ", loadBalancerComputedDeltas.getPassedContainerNames()), loadBalancerComputedDeltas.getDeltasToPass()),
                interleaved(format("Enable %s containers in load balancers", loadBalancerComputedDeltas.getAffectedServerNameForEnable()), loadBalancerComputedDeltas.getDeltasForEnable())
        );
    }

    protected LoadBalancerComputedDeltas computeDeltas(final DeltaSpecification specification) {
        List<Delta> deltas = specification.getDeltas();

        DeployedApplication deployedApplication = specification.getDeployedApplication();
        Environment targetEnvironment = deployedApplication.getEnvironment();
        Set<LoadBalancer> loadBalancers = getMembersOfType(targetEnvironment, LOADBALANCER_TYPE);

        final Set<Container> containers = collectContainers(new Deltas(deltas), Container.class);

        ListMultimap<Container, LoadBalancer> affectedAppServers = getContainerToLoadBalancersMap(loadBalancers, containers);
        logger.debug("App Servers affected by the deployment: {}", affectedAppServers.keySet());
        ListMultimap<Container, LoadBalancer> affectedWebServers = getAffectedWebServers(loadBalancers, containers);
        logger.debug("Web Servers affected by the deployment: {}", affectedWebServers.keySet());
        ListMultimap<Container, LoadBalancer> allAffectedServers = ArrayListMultimap.create(affectedAppServers);
        allAffectedServers.putAll(affectedWebServers);

        List<Delta> deltasToPass = new ArrayList<>();
        ListMultimap<LoadBalancer, LoadBalancerDeployed> deployedsForDisablePerLoadBalancer = LinkedListMultimap.create();
        ListMultimap<LoadBalancer, LoadBalancerDeployed> deployedsForEnablePerLoadBalancer = LinkedListMultimap.create();

        for (Delta delta : deltas) {
            Container container = ContainerCollector.DELTA_CONTAINER_FUNCTION.apply(delta);
            Delta deltaToPass = delta;
            if (null != container) {
                List<LoadBalancer> affectedLoadBalancers = allAffectedServers.get(container);
                Deltas deltasForContainer = new Deltas(deltasForContainer(deltas, container));
                for (LoadBalancer loadBalancer : affectedLoadBalancers) {
                    LoadBalancerDeployed deployedForDisable = new DisableServerInLoadBalancer(loadBalancer, container, deltasForContainer);
                    deployedsForDisablePerLoadBalancer.put(loadBalancer, deployedForDisable);
                    LoadBalancerDeployed deployedForEnable = new EnableServerInLoadBalancer(loadBalancer, container, deltasForContainer);
                    deployedsForEnablePerLoadBalancer.put(loadBalancer, deployedForEnable);
                }
                // mark deltas as already processed by orchestrator so that contributor can skip them
                deltaToPass = markDeltaAsProcessed(delta);
            }
            deltasToPass.add(deltaToPass);
        }
        List<LoadBalancerDeployed> deployedsForDisable = reduceDeployedsPerLoadBalancer(deployedsForDisablePerLoadBalancer);
        List<LoadBalancerDeployed> deployedsForEnable = reduceDeployedsPerLoadBalancer(deployedsForEnablePerLoadBalancer);

        return new LoadBalancerComputedDeltas(deltasToPass, deployedsForDisable, deployedsForEnable);
    }

    // reduce number of deployeds per load balancer
    private List<LoadBalancerDeployed> reduceDeployedsPerLoadBalancer(ListMultimap<LoadBalancer, LoadBalancerDeployed> deployedsPerLoadBalancer) {
        List<LoadBalancerDeployed> result = new ArrayList<>();
        Map<LoadBalancer, Collection<LoadBalancerDeployed>> loadBalancerDeployeds = deployedsPerLoadBalancer.asMap();

        Set<LoadBalancer> lbs = loadBalancerDeployeds.keySet();
        // try to combine deployeds (this should have effect on netscaler):
        for (LoadBalancer loadBalancer : lbs) {
            Collection<LoadBalancerDeployed> deployeds = loadBalancerDeployeds.get(loadBalancer);
            if (null != deployeds) {
                result.add(combineDeployeds(deployeds));
            }
        }

        return result;
    }

    // combine deployeds that belong to the same load balancer into a single deployed
    private LoadBalancerDeployed combineDeployeds(Collection<LoadBalancerDeployed> deployeds) {
        LoadBalancerDeployed result = null;
        Iterator<LoadBalancerDeployed> iterator = deployeds.iterator();
        if (iterator.hasNext()) {
            result = iterator.next();
            while (iterator.hasNext()) {
                LoadBalancerDeployed next = iterator.next();
                result.combine(next);
            }
        }
        return result;
    }

    protected String descForAppDeployment(final DeployedApplication deployedApplication) {
        return format("Deploy application %s", deployedApplication.getName());
    }

    private Delta markDeltaAsProcessed(Delta delta) {
        return new ProcessedDelta(delta);
    }

    private List<Delta> deltasForContainer(final List<Delta> deltas, final Container container) {
        return newArrayList(from(deltas).filter(new Predicate<Delta>() {
            @Override
            public boolean apply(Delta delta) {
                Container deltaContainer = ContainerCollector.DELTA_CONTAINER_FUNCTION.apply(delta);
                return container.equals(deltaContainer);
            }
        }));
    }

    private static final Logger logger = LoggerFactory.getLogger(LoadBalancingOrchestrator.class);

}
