package com.xebialabs.deployit.plugin.lb.planning;

import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Predicate;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.SetMultimap;

import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Environment;
import com.xebialabs.deployit.plugin.lb.ci.LoadBalancer;
import com.xebialabs.deployit.plugin.lb.planning.orchestrator.ProcessedDelta;
import com.xebialabs.deployit.plugin.lb.util.LoadBalancedContainers.LoadBalancingBounds;

import static com.google.common.collect.FluentIterable.from;
import static com.xebialabs.deployit.plugin.lb.util.DeploymentGroups.getDeploymentGroup;
import static com.xebialabs.deployit.plugin.lb.util.Environments.getMembersOfType;
import static com.xebialabs.deployit.plugin.lb.util.LoadBalancedContainers.getAffectedWebServers;
import static com.xebialabs.deployit.plugin.lb.util.LoadBalancedContainers.getContainerToLoadBalancersMap;
import static com.xebialabs.deployit.plugin.lb.util.LoadBalancedContainers.getLoadBalancingBounds;

public class ManageLoadBalancerPools {
    private static final Type LOADBALANCER_TYPE = Type.valueOf(LoadBalancer.class);

    @Contributor
    public static void manageContainersInPool(Deltas deltas, DeploymentPlanningContext ctx) {
        // skip deltas already processed by orchestrator
        ImmutableList<Delta> nonProcessedDeltas = from(deltas.getDeltas()).filter(new Predicate<Delta>() {
            @Override
            public boolean apply(final Delta delta) {
                if (delta instanceof ProcessedDelta) {
                    return false;
                }
                return true;
            }
        }).toList();
        final Set<Container> containers = ContainerCollector.collectContainers(new Deltas(nonProcessedDeltas), Container.class);

        Environment targetEnvironment = ctx.getDeployedApplication().getEnvironment();
        Set<LoadBalancer> loadBalancers = getMembersOfType(targetEnvironment, LOADBALANCER_TYPE);
        if (loadBalancers.isEmpty()) {
            logger.debug("No load balancers in environment. Nothing to do.");
            return;
        }

        Integer currentDeploymentGroup = getCurrentDeploymentGroup(containers);
        if (currentDeploymentGroup == null) {
            logger.info("Unable to determine deployment group for containers: {}. Will not add steps.", containers);
            return;
        }

        // container -> load balancers managing that server
        ListMultimap<Container, LoadBalancer> affectedAppServers = getContainerToLoadBalancersMap(loadBalancers, containers);
        logger.debug("App Servers affected by the deployment: {}", affectedAppServers.keySet());
        ListMultimap<Container, LoadBalancer> affectedWebServers = getAffectedWebServers(loadBalancers, containers);
        logger.debug("Web Servers affected by the deployment: {}", affectedWebServers.keySet());

        ListMultimap<Container, LoadBalancer> allAffectedServers = ArrayListMultimap.create(affectedAppServers);
        allAffectedServers.putAll(affectedWebServers);

        // container -> first/last deployment groups of the fronted servers
        Map<Container, LoadBalancingBounds> containerBalancingBounds = getLoadBalancingBounds(allAffectedServers.keySet());

        SetMultimap<LoadBalancer, Container> stopTrafficNowOn = HashMultimap.create();
        SetMultimap<LoadBalancer, Container> startTrafficNowOn = HashMultimap.create();

        for (Entry<Container, LoadBalancingBounds> balancingBounds : containerBalancingBounds.entrySet()) {
            Container container = balancingBounds.getKey();
            LoadBalancingBounds bounds = balancingBounds.getValue();
            List<LoadBalancer> balancers = allAffectedServers.get(container);
            if (bounds.startGroup == currentDeploymentGroup) {
                addBalancedContainers(stopTrafficNowOn, balancers, container);
            }
            if (bounds.endGroup == currentDeploymentGroup) {
                addBalancedContainers(startTrafficNowOn, balancers, container);
            }
        }

        for (LoadBalancer loadBalancer : stopTrafficNowOn.keySet()) {
            Set<Container> serversToDisable = stopTrafficNowOn.get(loadBalancer);
            logger.debug("Deployment group [{}]: Adding steps to *stop* traffic to [{}] on loadbalancer [{}]", currentDeploymentGroup, serversToDisable, loadBalancer);
            loadBalancer.stopTraffic(deltas, ctx, serversToDisable);
        }

        for (LoadBalancer loadBalancer : startTrafficNowOn.keySet()) {
            Set<Container> serversToEnable = startTrafficNowOn.get(loadBalancer);
            logger.debug("Deployment group [{}]: Adding steps to *start* traffic to [{}] on loadbalancer [{}]", currentDeploymentGroup, serversToEnable, loadBalancer);
            loadBalancer.startTraffic(deltas, ctx, serversToEnable);
        }
    }


    private static void addBalancedContainers(Multimap<LoadBalancer, Container> balancerToContainers, List<LoadBalancer> balancers, Container container) {
        for (LoadBalancer balancer : balancers) {
            balancerToContainers.put(balancer, container);
        }
    }

    /**
     * The deployment group that the contributor is currently processing. This can be found by looking at the deltas
     * currently being processed - the group-based orchestrator will only include a delta in the current set if its
     * container is in the current deployment group.
     *
     * So by looking at the deployment groups of the containers being processed we can find the current group (if
     * the containers have no group, we are in the 'unallocated' group).

     * @param containers
     * @return
     */
    private static Integer getCurrentDeploymentGroup(Set<Container> containers) {
        // expecting at least one target container
        if (!containers.isEmpty()) {
            return getDeploymentGroup(containers.iterator().next());
        }
        return null;
    }

    private static final Logger logger = LoggerFactory.getLogger(ManageLoadBalancerPools.class);
}
