package com.xebialabs.deployit.plugin.jbossdm.step;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Preview;
import com.xebialabs.deployit.plugin.api.flow.PreviewStep;
import com.xebialabs.deployit.plugin.api.flow.StageableStep;
import com.xebialabs.deployit.plugin.api.flow.StagedFile;
import com.xebialabs.deployit.plugin.api.flow.StagingContext;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.rules.RulePostConstruct;
import com.xebialabs.deployit.plugin.api.rules.StepMetadata;
import com.xebialabs.deployit.plugin.api.rules.StepParameter;
import com.xebialabs.deployit.plugin.api.rules.StepPostConstructContext;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.jbossdm.container.CliManagedContainer;
import com.xebialabs.deployit.plugin.steps.CalculatedStep;
import com.xebialabs.deployit.plugin.steps.ContextHelper;
import com.xebialabs.deployit.plugin.steps.TargetContainerJavaHelper;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;

import static com.xebialabs.deployit.plugin.remoting.vars.VarsConverter.STAGED_FILE_VARIABLE_NAME_PREFIX;

@StepMetadata(name = "jboss-cli")
public class JbossCliStep extends CalculatedStep implements PreviewStep, StageableStep {
    protected static final Logger logger = LoggerFactory.getLogger(JbossCliStep.class);

    @StepParameter(description = "Path to the python script to be executed.")
    private String script;

    @StepParameter(description = "Dictionary that represent context available to the python script", required = false, calculated = true)
    private Map<String, Object> pythonContext = new HashMap<>();

    @StepParameter(description = "CLI managed container (domain, standalone server, profile or managed group) where to execute the cli script",
        calculated = true)
    private CliManagedContainer container;

    @StepParameter(description = "List of python library scripts that should be automatically loaded when using a JBoss CLI script. Usage of additional script libraries is discouraged, please use proper Python modules.", required = false, calculated = true)
    private List<String> additionalLibraries = new ArrayList<>();

    @StepParameter(
        description = "If true, artifacts from the python context will be uploaded to the target host and available to the script as <bindingName>.file (of type OverthereFile). Defaults to true. If set to false it will prevent staging.")
    private Boolean uploadArtifactsInPythonContext = true;

    private BaseStep cliStep;

    @RulePostConstruct
    @Override
    public void doPostConstruct(StepPostConstructContext ctx) {
        super.doPostConstruct(ctx);
        pythonContext = ContextHelper.defaultContext(ctx, pythonContext);
        container = calculateTargetContainer(ctx);
        cliStep = new CliDeploymentStep(script, getOrder(), pythonContext, getDescription(), container);
        cliStep.setAdditionalLibraries(additionalLibraries);
    }

    private CliManagedContainer calculateTargetContainer(StepPostConstructContext ctx) {
        CliManagedContainer targetcontainer = container;
        if (null == targetcontainer) {
            targetcontainer = TargetContainerJavaHelper.defaultTargetContainer(ctx, CliManagedContainer.class);
        }
        return targetcontainer;
    }

    @Override
    public Preview getPreview() {
        return cliStep.getPreview();
    }

    @Override
    public void requestStaging(StagingContext ctx) {
        if (!uploadArtifactsInPythonContext) {
            return;
        }

        for (Map.Entry<String, Object> entry : pythonContext.entrySet()) {
            Object entryVal = entry.getValue();
            if (entryVal instanceof Artifact) {
                Artifact artifact = (Artifact) entryVal;
                StagedFile stagedFile = ctx.stageArtifact(artifact, container.getHost());
                pythonContext.put(entry.getKey() + STAGED_FILE_VARIABLE_NAME_PREFIX, stagedFile);
            }
        }
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        StepExitCode exitCode = StepExitCode.FAIL;
        try (OverthereConnection remoteConnection = cliStep.getRemoteConnection()) {
            CliVarsConverter converter = new CliVarsConverter(ctx, remoteConnection, pythonContext, uploadArtifactsInPythonContext);
            converter.convert();
            exitCode = cliStep.execute(ctx);
        }

        return exitCode;
    }

    private static class CliVarsConverter {
        private Map<String, Object> pythonContext;
        private ExecutionContext ctx;
        private OverthereConnection remoteConnection;

        private boolean uploadArtifactsInPythonContext;

        private CliVarsConverter(ExecutionContext ctx, OverthereConnection remoteConnection, Map<String, Object> pythonContext,
            boolean uploadArtifactsInPythonContext) {
            this.ctx = ctx;
            this.remoteConnection = remoteConnection;
            this.pythonContext = pythonContext;
            this.uploadArtifactsInPythonContext = uploadArtifactsInPythonContext;
        }

        private void convert() {
            TreeSet<String> sortedKeys = new TreeSet<>(pythonContext.keySet());
            for (String varName : sortedKeys) {
                Object varValue = pythonContext.get(varName);
                logger.debug("Converting variable [{}]", varName);
                convertVariable(varName, varValue);
            }
        }

        private void convertVariable(String varName, Object varValue) {
            if (varValue instanceof Artifact) {
                convertArtifact(varName, (Artifact) varValue);
            }
        }

        /**
         * Since python scripts are executed locally, but scripts interact with the CLI on remote system users might expect
         * to have access to automatically uploaded artifacts on a target container.
         *
         * ArtifactVarsConverts scans variables of type Artifact in python context and replaces 'file' property with staged
         * or automatically uploaded overthere file.
         *
         * This means that 'deployed.file' returns OverthereFile on a remote container - which is suitable
         * because it doesn't have to be uploaded to working directory explicitly.
         *
         * One can use uploadArtifactsInPythonContext boolean flag to turn off automatic upload and replacement of file property.
         */
        private void convertArtifact(String varName, Artifact artifact) {
            if (!uploadArtifactsInPythonContext) {
                logger.debug("Property 'file' of [{}] will NOT be automatically uploaded and adjusted to point to a remote overthere file.", varName);
                return;
            }

            final String stagedFileVarName = varName + STAGED_FILE_VARIABLE_NAME_PREFIX;
            if (ctx != null && pythonContext.containsKey(stagedFileVarName)) {
                StagedFile stagedFileReference = (StagedFile) pythonContext.get(stagedFileVarName);
                // replace file property with staged artifact file
                OverthereFile stagedOverthereFile = stagedFileReference.get(remoteConnection, ctx);
                logger.debug("Setting file property of [{}] to an already staged file [{}]", varName, stagedOverthereFile.getPath());
                artifact.setFile(stagedOverthereFile);
                return;
            }

            if (artifact.getFile() == null) {
                logger.debug("Setting file property of [{}] to null because artifact has a null file reference.", varName);
                return;
            }

            logger.debug("Creating temporary file for variable [{}]", varName);
            OverthereFile uploadedFileArtifact = remoteConnection.getTempFile(artifact.getFile().getName());
            logger.debug("Uploading artifact for variable [{}] to [{}]", varName, uploadedFileArtifact.getPath());
            artifact.getFile().copyTo(uploadedFileArtifact);
            logger.debug("Uploaded artifact for variable [{}] to [{}]", varName, uploadedFileArtifact.getPath());
            // replace file property with uploaded artifact file
            artifact.setFile(uploadedFileArtifact);
        }
    }
}
