package com.xebialabs.deployit.plugin.jbossdm.step;

import java.io.*;
import java.net.URL;
import java.util.List;
import java.util.Map;

import javax.script.*;

import org.python.core.Options;
import org.python.core.PyException;
import org.python.core.PySystemState;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Charsets;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.io.ByteStreams;
import com.google.common.io.Resources;

import com.xebialabs.deployit.plugin.api.flow.*;
import com.xebialabs.deployit.plugin.jbossdm.container.CliManagedContainer;
import com.xebialabs.deployit.plugin.jbossdm.exception.CliScriptException;
import com.xebialabs.deployit.plugin.jbossdm.exception.CliScriptExit;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.RuntimeIOException;
import com.xebialabs.overthere.local.LocalConnection;
import com.xebialabs.overthere.local.LocalFile;
import com.xebialabs.overthere.util.OverthereUtils;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Strings.emptyToNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.overthere.util.OverthereUtils.closeQuietly;

@SuppressWarnings("serial")
public abstract class BaseStep implements PreviewStep {

    public static final String JYTHON_SCRIPT_ENGINE = "jython";

    protected final Logger logger = LoggerFactory.getLogger(this.getClass());

    private final String script;
    private final int order;
    private final Map<String, Object> pythonContext;
    private String description;
    private CliManagedContainer container;
    private List<String> additionalLibraries = newArrayList();

    private String remoteWorkingDirPath;
    private boolean retainRemoteWorkingDirOnCompletion;

    private transient OverthereConnection localConn;
    private transient OverthereConnection remoteConn;
    private transient ExecutionContext ctx;
    private transient OverthereFile remoteWorkingDir;


    protected BaseStep(String script, int order, Map<String, Object> pythonContext, String description, CliManagedContainer container) {
        this.script = script;
        this.order = order;
        this.pythonContext = pythonContext;
        this.description = description;
        this.container = container;

    }

    @Override
    public int getOrder() {
        return order;
    }

    @Override
    public Preview getPreview() {
        try {
            StringBuilder builder = new StringBuilder();
            if (!additionalLibraries.isEmpty()) {
                for (String library : additionalLibraries) {
                    String script = Resources.toString(Resources.getResource(library), Charsets.UTF_8);
                    checkNotNull(script, "Library {} cannot be found on class path.", library);
                    builder.append("\n\n#").append(library).append("\n").append(script);
                }
            }

            String scriptContent = Resources.toString(Resources.getResource(script), Charsets.UTF_8);
            builder.append("\n\n#").append(script).append("\n").append(scriptContent);
            return Preview.withSourcePathAndContents(script, builder.toString());
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    @Override
    public String getDescription() {
        return description;
    }

    public StepExitCode handleExecute(ExecutionContext ctx) throws Exception {
        try {
            this.ctx = ctx;
            ScriptEngine scriptEngine = loadScriptEngine();
            executeScript(scriptEngine);
            return StepExitCode.SUCCESS;
        } finally {
            disconnect();
        }
    }

    protected void executeScript(ScriptEngine scriptEngine) throws IOException {
        Bindings bindings = createBindings(pythonContext);
        bindings.put("step", this);
        loadLibraryScriptsAndEval(script, scriptEngine, bindings);
    }

    protected void loadLibraryScriptsAndEval(String scriptPath, ScriptEngine scriptEngine, Bindings localBindings) throws IOException {
        String script = Resources.toString(Resources.getResource(scriptPath), Charsets.UTF_8);
        //need to do this in order to work around a bug in the python script engine, which does not use the bindings
        //to compile the script when using the method scriptEngine.eval(String,Bindings).
        Bindings origEngineBindings = scriptEngine.getBindings(ScriptContext.ENGINE_SCOPE);
        try {
            Bindings engineAndLocalScope = new SimpleBindings();
            engineAndLocalScope.putAll(origEngineBindings);
            engineAndLocalScope.putAll(localBindings);
            scriptEngine.setBindings(engineAndLocalScope, ScriptContext.ENGINE_SCOPE);
            loadLibraryScripts(additionalLibraries, scriptEngine);
            logger.debug("Executing script " + scriptPath);
            if (logger.isTraceEnabled()) {
                logger.trace(script);
            }
            scriptEngine.eval(script);
        } catch (ScriptException e) {
            throwCliScriptException(scriptPath, e);
        } finally {
            scriptEngine.setBindings(origEngineBindings, ScriptContext.ENGINE_SCOPE);
        }
    }

    protected ScriptEngine loadScriptEngine() throws IOException {
        Options.includeJavaStackInExceptions = false;
        PySystemState engineState = new PySystemState();

        ScriptEngine scriptEngine = new ScriptEngineManager().getEngineByName(JYTHON_SCRIPT_ENGINE);
        checkNotNull(scriptEngine, "Jython Script Engine cannot be initialized. Make sure jython jars are on the class path.");
        loadLibraryScripts(container.getManagingContainer().getLibraries(), scriptEngine);
        return scriptEngine;
    }

    protected void loadLibraryScripts(List<String> libs, ScriptEngine scriptEngine) throws IOException {
        if (!libs.isEmpty()) {
            for (String library : libs) {
                String script = Resources.toString(Resources.getResource(library), Charsets.UTF_8);
                checkNotNull(script, "Library {} cannot be found on class path.", library);
                try {
                    scriptEngine.eval(script);
                } catch (ScriptException e) {
                    throwCliScriptException(library, e);
                }
            }
        }
    }

    protected void throwCliScriptException(String scriptName, ScriptException e) {
        if (e.getCause() instanceof PyException) {
            handlePySyntaxError(scriptName, e);
        } else if (e.getMessage().startsWith(CliScriptExit.class.getName())) {
            handleCliScriptExit(scriptName, e);
        } else if (e.getMessage().startsWith("AttributeError:")) {
            handleAttributeError(scriptName, e);
        }
        throw new CliScriptException(scriptName, e);
    }

    private void handleCliScriptExit(String scriptName, ScriptException e) {
        String msg = e.getMessage().replaceAll(CliScriptExit.class.getName() + ":", "");
        msg = msg.replaceFirst("<script>", scriptName);
        if (e.getCause() instanceof PyException) {
            String stack = extractPyStack(scriptName, (PyException) e.getCause());
            throw new CliScriptExit(msg + "\n\n" + stack);
        }
        throw new CliScriptExit(msg);

    }

    private void handleAttributeError(String scriptName, ScriptException e) {
        String msg = e.getMessage();
        msg = msg.replaceFirst("<script>", scriptName);
        throw new CliScriptException(msg);
    }

    private void handlePySyntaxError(String scriptName, ScriptException e) {
        String msg = e.getCause().toString();
        msg = msg.replaceFirst("\"<script>\"", "\"" + scriptName + "\"");
        throw new CliScriptException(msg);
    }

    private String extractPyStack(final String scriptName, PyException pyException) {
        StringBuilder stackBuffer = new StringBuilder();
        pyException.traceback.dumpStack(stackBuffer);
        String stack = stackBuffer.toString();
        stack = stack.replaceFirst("<script>", scriptName);
        return stack;
    }

    protected Bindings createBindings(Map<String, Object> variables) {
        Bindings bindings = new SimpleBindings();
        bindings.putAll(variables);
        return bindings;
    }

    public Object executeCliCommand(String cmd) {
        return container.getManagingContainer().execute(ctx, cmd);
    }

    public List<Object> executeCliCommands(String[] cmds) {
        return container.getManagingContainer().execute(ctx, cmds);
    }

    public ExecutionContext getCtx() {
        return ctx;
    }

    public void setAdditionalLibraries(List<String> additionalLibraries) {
        this.additionalLibraries = additionalLibraries;
    }

    protected List<String> getAdditionalLibraries() {
        return additionalLibraries;
    }

    public CliManagedContainer getContainer() {
        return container;
    }

    public OverthereFile getRemoteWorkingDirectory() {
        if (remoteWorkingDir == null) {
            OverthereFile tempDir;
            if (Strings.isNullOrEmpty(getRemoteWorkingDirPath())) {
                tempDir = getRemoteConnection().getTempFile("jbossdm_plugin", ".tmp");
            } else {
                tempDir = getRemoteConnection().getFile(getRemoteWorkingDirPath());
            }
            tempDir.mkdir();
            remoteWorkingDir = tempDir;
        }
        return remoteWorkingDir;
    }


    public OverthereConnection getLocalConnection() {
        if (localConn == null) {
            localConn = LocalConnection.getLocalConnection();
        }
        return localConn;
    }

    public OverthereConnection getRemoteConnection() {
        if (remoteConn == null) {
            remoteConn = container.getManagingContainer().getHost().getConnection();
        }
        return remoteConn;
    }

    protected void disconnect() {
        if (localConn != null) {
            closeQuietly(localConn);
        }

        if (!Strings.isNullOrEmpty(getRemoteWorkingDirPath()) && !isRetainRemoteWorkingDirOnCompletion()) {
            getRemoteWorkingDirectory().deleteRecursively();
        }

        if (remoteConn != null) {
            closeQuietly(remoteConn);
        }

        remoteWorkingDir = null;
        localConn = null;
        remoteConn = null;
    }

    public String getRemoteWorkingDirPath() {
        return remoteWorkingDirPath;
    }

    public void setRemoteWorkingDirPath(String remoteWorkingDirPath) {
        this.remoteWorkingDirPath = remoteWorkingDirPath;
    }

    public boolean isRetainRemoteWorkingDirOnCompletion() {
        return retainRemoteWorkingDirOnCompletion;
    }

    public void setRetainRemoteWorkingDirOnCompletion(boolean deleteWorkingDirOnCompletion) {
        this.retainRemoteWorkingDirOnCompletion = deleteWorkingDirOnCompletion;
    }

    public OverthereFile uploadToWorkingDirectory(String content, String fileName) {
        getCtx().logOutput("Uploading file " + fileName + " to working directory.");
        OverthereFile target = getRemoteWorkingDirectory().getFile(fileName);
        OverthereUtils.write(content.getBytes(), target);
        return target;
    }

    public OverthereFile uploadToWorkingDirectory(File content, String fileName) {
        return uploadToWorkingDirectory(LocalFile.valueOf(content), fileName);
    }

    public OverthereFile uploadToWorkingDirectory(OverthereFile content, String fileName) {
        String fileType = content.isDirectory() ? "directory" : "file";
        getCtx().logOutput("Uploading " + fileType + " " + fileName + " to working directory.");
        OverthereFile target = getRemoteWorkingDirectory().getFile(fileName);
        content.copyTo(target);
        return target;
    }

    public OverthereFile uploadToWorkingDirectory(URL content, String fileName) {
        getCtx().logOutput("Uploading file " + fileName + " to working directory.");
        OverthereFile target = getRemoteWorkingDirectory().getFile(fileName);
        OutputStream out = target.getOutputStream();
        try {
            Resources.copy(content, out);
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        } finally {
            closeQuietly(out);
        }
        return target;
    }

    public boolean hostFileExists(String remoteFile) {
        checkNotNull(emptyToNull(remoteFile));
        OverthereFile file = getRemoteConnection().getFile(remoteFile);
        return file.exists();
    }


    public String readHostFile(final String remoteFile) {
        checkNotNull(emptyToNull(remoteFile));
        OverthereFile file = getRemoteConnection().getFile(remoteFile);
        checkArgument(file.exists(), "File %s does not exist on host %s", remoteFile, getContainer().getManagingContainer().getHost());
        try {
            InputStream in = file.getInputStream();
            try {
                byte[] bytes = ByteStreams.toByteArray(in);
                return new String(bytes);
            } finally {
                closeQuietly(in);
            }
        } catch (IOException e) {
            throw new RuntimeIOException("Failed to read file " + remoteFile, e);
        }
    }

    public String[] readHostFileLines(final String remoteFile) {
        String data = readHostFile(remoteFile);
        Iterable<String> iterable = Splitter.on(getContainer().getManagingContainer().getHost().getOs().getLineSeparator()).split(data);
        return Iterables.toArray(iterable, String.class);
    }

}
