package com.xebialabs.deployit.plugin.jbossdm.step;

import com.google.common.io.Closeables;
import com.xebialabs.deployit.plugin.api.execution.ExecutionContext;
import com.xebialabs.deployit.plugin.api.execution.ExecutionContextListener;
import com.xebialabs.deployit.plugin.api.utils.Strings;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereProcess;
import com.xebialabs.overthere.RuntimeIOException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.util.Timer;
import java.util.TimerTask;

public class CliDaemon implements ExecutionContextListener {

    private String cliExecutable;
    private final String username;
    private final String password;
    private final String adminHost;
    private final int adminPort;
    private final Host connectingHost;
    private ExecutionContext context;

    private OverthereConnection connection;
    private OverthereProcess process;

    private static final int FLUSH_DELAY_MS = 5000;
    private static final int FLUSH_CHECK_INTERVAL_MS = 2000;

    private static final Timer flushTimer = new Timer("CliDaemon-AutoFlushTimer", true);

    public CliDaemon(String cliExecutable, String username, String password, String adminHost, int adminPort, Host connectingHost, ExecutionContext context) {
        this.cliExecutable = cliExecutable;
        this.username = username;
        this.password = password;
        this.adminHost = adminHost;
        this.adminPort = adminPort;
        this.connectingHost = connectingHost;
        this.context = context;
    }

    public Object executeCliCommandWithDaemon(String cmd) {
        if (connection == null) {
            connection = connectingHost.getConnection();
        }
        CmdLine cmdLine = createBasicCliCmdLine();
        cmdLine.addArgument("--command=" + cmd);
        CliProcessOutputHandler handler = new CliProcessOutputHandler();
        int rc = connection.execute(handler, cmdLine);
        if (rc != 0) {
            throw new RuntimeIOException("Cli command failed with return code " + rc + ". Details : " + handler.getResult());
        }
        return handler.getResult();
    }

    public Object executeCliCommand(String cmd)  {
        if (!isConnected()) {
            connect();
        }

        CliProcessOutputHandler handler = new CliProcessOutputHandler();
        executeCommand(handler, cmd);
        return handler.getResult();
    }

    private boolean isConnected() {
        return connection != null && process != null;
    }

    public void disconnect() {
        if (process != null) {
            process.destroy();
			process = null;
        }

        if (connection != null) {
            connection.close();
			connection = null;
        }
    }

    public void connect() {
        disconnect();
        connection = connectingHost.getConnection();
        CmdLine cmdLine = createBasicCliCmdLine();

        process = connection.startProcess(cmdLine);
        waitForCliStart();
    }

    private CmdLine createBasicCliCmdLine() {
        CmdLine cmdLine = new CmdLine();
        cmdLine.addArgument(cliExecutable).addArgument("--connect").addArgument("--controller="+adminHost+":"+adminPort);
        if (!Strings.isBlank(username)) {
            cmdLine.addArgument("--user=" + username);
        }
        if (!Strings.isBlank(password)) {
            cmdLine.addPassword("--password=" + password);
        }
        return cmdLine;
    }

    protected void waitForCliStart() {
        final InputStreamReader stdout = new InputStreamReader(process.getStdout());
        final StringBuilder stdoutLineBuffer = new StringBuilder();
        final long[] flushAfter = new long[1];
        final TimerTask flushTimerTask = new TimerTask() {
            @Override
            public void run() {
                synchronized (stdoutLineBuffer) {
                    if (flushAfter[0] < System.currentTimeMillis()) {
                        if (stdoutLineBuffer.length() > 0) {
                            context.logOutput(stdoutLineBuffer.toString());
                            stdoutLineBuffer.setLength(0);
                        }
                        flushAfter[0] = System.currentTimeMillis() + FLUSH_DELAY_MS;
                    }
                }
            }
        };
        flushTimer.schedule(flushTimerTask, FLUSH_DELAY_MS, FLUSH_CHECK_INTERVAL_MS);
        try {
            for (;;) {
                try {
                    int cInt = stdout.read();   //TODO : Use something like SimpleTimerLimiter from guava to fail after 10 seconds or so.
                    if (cInt == -1) {
                        captureStderr(context, process);
                        int exitCode = -1;
                        try {
                            exitCode = process.waitFor();
                        } catch (InterruptedException exc) {
                            logger.error("Interrupted while waiting for " + process + " to complete");
                            Thread.currentThread().interrupt();
                        }
                        throw new RuntimeIOException("Cannot start cli: exit code " + exitCode);
                    } else {
                        char c = (char) cInt;
                        if (c != '\r' && c != '\n') {
                            synchronized (stdoutLineBuffer) {
                                stdoutLineBuffer.append(c);
                            }
                        }
                        if (c == ']') {
                            synchronized (stdoutLineBuffer) {
                                flushAfter[0] = System.currentTimeMillis() + FLUSH_DELAY_MS;
                                String stdoutLine = stdoutLineBuffer.toString();
                                stdoutLineBuffer.setLength(0);
                                if (stdoutLine.trim().matches("\\[(domain|standalone)@.+:[0-9]+./\\]")) {
                                    break;
                                }
                                context.logOutput(stdoutLine);
                            }
                        }
                    }
                } catch (IOException exc) {
                    throw new RuntimeIOException("Cannot start cli", exc);
                }
            }
        } finally {
            flushTimerTask.cancel();
        }
    }

    private void captureStderr(final ExecutionContext context, OverthereProcess daemonProcess) throws IOException {
        final BufferedReader stderr = new BufferedReader(new InputStreamReader(daemonProcess.getStderr()));
        for (;;) {
            final String stderrLine = stderr.readLine();
            if (stderrLine == null) {
                break;
            }
            context.logError(stderrLine);
        }
    }

    private void executeCommand(CliProcessOutputHandler handler, String command) {
        try {
            logger.info("Executing command {} on {} (with cli daemon)", command, connection);
            if (command.charAt(command.length()-1) == '\n')
                command = command.substring(0, command.length()-1);

            String daemonLine = command + "\n";
            OutputStream stdin = process.getStdin();
			stdin.write(daemonLine.getBytes());
            stdin.flush();

            InputStreamReader stdout = new InputStreamReader(process.getStdout());
            StringBuilder lineBuffer = new StringBuilder();
            for (;;) {
                char c = (char)stdout.read();
                if (c == '\r') {
                    continue;
                }
                String line = lineBuffer.toString();
                if (c != '\n') {
                    lineBuffer.append(c);
                } else {
                    if (!line.equals(command)) {
                        handler.handleOutputLine(line);

                    }
                    lineBuffer = new StringBuilder();
                    continue;
                }

                if (line.trim().matches("\\[(domain|standalone)@.+:[0-9]+\\s.*\\]")) {
                    //command complete
                    break;
                } else {
                    handler.handleOutput(c);
                }
            }
        } catch (IOException exc) {
            throw new RuntimeIOException("Cannot execute command " + command + " on " + connectingHost, exc);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(CliDaemon.class);

	@Override
	public void contextDestroyed() {
		disconnect();
	}
}
