# Get provided application pool name or use the deployed name.
$applicationPoolName = if($deployed.applicationPoolName) { $deployed.applicationPoolName } else { $deployed.name }

# Verify application pool name is not being changed.
if($previousDeployed) {
    $previousApplicationPoolName = if($previousDeployed.applicationPoolName) { $previousDeployed.applicationPoolName } else { $previousDeployed.name }
    if($applicationPoolName -ne $previousApplicationPoolName) {
        Write-Host "Renaming an application pool is not supported. Undeploy and deploy the application pool instead."
        Exit 1
    }
}

# Verify username and password are provided if serviceAccount if set to 'SpecificUser'.
if ($deployed.serviceAccount -eq "SpecificUser") {
    if (!$deployed.username) {
        throw "Username is required when service account is [SpecificUser]."
    }
    if (!$deployed.password) {
        throw "Password is required when service account is [SpecificUser]."
    }
}

# Check whether application pool already exists and then either retrieve or create it.
$appPoolPath = "IIS:\AppPools\$applicationPoolName"
if (Test-Path $appPoolPath) {
    Write-Host "Modifying existing application pool [$applicationPoolName]."

    $applicationPool = Get-Item $appPoolPath
} else {
    Write-Host "Creating new application pool [$applicationPoolName]."

    $applicationPool = New-Item $appPoolPath

    # Sleep after creating the application pool, IIS is sometimes not ready to stop it immediately
    Write-Host "Waiting for application pool [$applicationPoolName] to initialize."
    Start-Sleep -Seconds $deployed.sleepAfterCreate

    # Stopping application pool so that we can configure it correctly before starting it properly from start-application-pool.ps1
    Stop-WebAppPool $applicationPoolName
}

# Set application pool properties.
$applicationPool.managedRuntimeVersion = $deployed.managedRuntimeVersion
$applicationPool.managedPipelineMode = if ($deployed.useClassicManagedPipeline) { 'Classic' } else { 'Integrated' }
$applicationPool.enable32BitAppOnWin64 = $deployed.enable32BitAppOnWin64
$applicationPool.queueLength = $deployed.queueLength -as [int]

# Process Model settings
$applicationPool.ProcessModel.IdentityType = $deployed.serviceAccount
if ($deployed.serviceAccount -eq "SpecificUser") {
    $applicationPool.ProcessModel.UserName = $deployed.username
    $applicationPool.ProcessModel.Password = $deployed.password
}
$applicationPool.ProcessModel.PingInterval = (New-TimeSpan -sec $deployed.processModelPingInterval).toString()
$applicationPool.ProcessModel.PingResponseTime = (New-TimeSpan -sec $deployed.processModelPingResponseTime).toString()
$applicationPool.ProcessModel.StartupTimeLimit =  (New-TimeSpan -sec $deployed.processModelStartupTimeLimit).toString()
$applicationPool.ProcessModel.ShutdownTimeLimit = (New-TimeSpan -sec $deployed.processModelShutdownTimeLimit).toString()
$applicationPool.ProcessModel.MaxProcesses = $deployed.processModelMaxProcesses -as [int]
$applicationPool.ProcessModel.IdleTimeout = (New-TimeSpan -min $deployed.processModelIdleTimeout).toString()
$applicationPool.ProcessModel.pingingEnabled = $deployed.processModelPingEnabled -as [bool]

# Failure settings
$applicationPool.Failure.RapidFailProtection = $deployed.failureRapidFailProtection -as [bool]
$applicationPool.Failure.RapidFailProtectionInterval = (New-TimeSpan -min $deployed.failureRapidFailProtectionInterval).toString()
$applicationPool.Failure.RapidFailProtectionMaxCrashes = $deployed.failureRapidFailProtectionMaxCrashes -as [int]

# Recycling settings
$applicationPool.Recycling.PeriodicRestart.Time = (New-TimeSpan -min $deployed.recyclingPeriodicRestartTime).toString()

# Save application pool.
$applicationPool | Set-Item

#Periodic restart setting
clear-ItemProperty $appPoolPath -Name Recycling.periodicRestart.schedule #clear values
$deployed.recyclingPeriodicRestartSchedule | foreach {
    New-ItemProperty $appPoolPath -Name Recycling.periodicRestart.schedule -Value @{value=([TimeSpan]::Parse( $_)).toString()}
}

