# Get provided website name or use the deployed name.
$websiteName = if($deployed.websiteName) { $deployed.websiteName } else { $deployed.name }

# Verify website name is not being changed.
if($previousDeployed) {
    $previousWebsiteName = if($previousDeployed.websiteName) { $previousDeployed.websiteName } else { $previousDeployed.name }
    if($websiteName -ne $previousWebsiteName) {
        Write-Host "Renaming a website is not supported. Undeploy and deploy the website instead."
        Exit 1
    }
}

# Verify the website has at least one binding specified, otherwise it won't be able to start later
if(!$deployed.bindings){
    Write-Host "Website must have at least one binding specified."
    Exit 1
}

# check if https bindings contain certificate
foreach ($binding in $deployed.bindings) {
    if( "$($binding.protocol)" -eq "https" ) {
       if (!$($binding.certificateName)) {
           Write-Host "When an https binding is used, a certificatename must be specified."
           Exit 1
       }
    }
}

# Create physical path if it doesn't exist yet.
if (!(Test-Path $deployed.physicalPath)) {
    Write-Host "Creating empty website physical path [$($deployed.physicalPath)]."
    if(!(New-Item $deployed.physicalPath -Type Directory -Force)) {
        Write-Host "Cannot create directory [$($deployed.physicalPath)]."
        Exit 1
    }
}

# Check whether website already exists and then either modify or create it.
$websitePath = "IIS:\Sites\$websiteName"
if (Test-Path $websitePath) {
    Write-Host "Modifying existing website [$websiteName]."
 
    Set-ItemProperty -Path $websitePath -Name PhysicalPath -Value $deployed.physicalPath
} else {
    Write-Host "Creating new website [$websiteName]."

    $params = @{ Name=$websiteName; PhysicalPath=$deployed.physicalPath }

    if (!(Get-Website)) {
        Write-Host "INFO: Activating workaround for PowerShell bug that occurs when there are no websites yet."
        $params.id = 1
    }

    New-Website @params | Out-Null
}

# Set application pool.
if($deployed.applicationPoolName) {
    Set-ItemProperty -Path $websitePath -Name ApplicationPool -Value $deployed.applicationPoolName
}

# Remove existing bindings.
foreach ($existingBinding in (Get-WebBinding -Name $websiteName)) {
    Write-Host "Removing existing binding [$($existingBinding.bindingInformation)] from website [$websiteName]."

    $bindingProtocol = $($existingBinding.protocol)

    if( "$bindingProtocol" -eq "https" )
    {
        $bindingAdress = $($existingBinding.bindingInformation).split(':')[0]
        $bindingPort = $($existingBinding.bindingInformation).split(':')[1]
        if( "$bindingAdress" -eq "*" )
        {
            $bindingAdress = "0.0.0.0"
        }
        Write-Host "Removing related SslBinding [$($existingBinding.bindingInformation)] from website [$websiteName]"
        remove-item -path IIS:\SslBindings\$bindingAdress!$bindingPort
    }

    Remove-WebBinding -InputObject $existingBinding
}

# Add bindings.
foreach ($binding in $deployed.bindings) {
    Write-Host "Adding binding [$($binding.name)] for website [$websiteName]."

    $params = @{ Name=$websiteName; Protocol=$binding.protocol; Port=$binding.port; IPAddress=$binding.ipAddress }

    if ($binding.hostHeader) {
        $params.hostHeader = $binding.hostHeader
    }

    New-WebBinding @params | Out-Null

   if( "$($binding.protocol)" -eq "https" ) {

        $bindingPort = $($binding.port)
        $bindingAdress = $($binding.ipAddress)
        if( "$bindingAdress" -eq "*" )
        {
            $bindingAdress = "0.0.0.0"
        }

        $isSslBindingExists = Test-Path IIS:\SslBindings\$bindingAdress!$bindingPort
        if( $isSslBindingExists )
        {
            Write-Host "Removing existing SslBinding [$($binding.name)] for website [$websiteName]."
            remove-item -path IIS:\SslBindings\$bindingAdress!$bindingPort
        }

        Write-Host "Adding related SslBinding [$($binding.name)] for website [$websiteName]."
        get-item cert:\LocalMachine\My\$($binding.certificateName) | new-item -path IIS:\SslBindings\$bindingAdress!$bindingPort
    }
}

setAuthentication $deployed $websiteName
setDirectoryBrowsing $deployed $websiteName

