#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

Function getWebsite($websiteName) {
    Return Get-WmiObject -Namespace "ROOT\MicrosoftIISv2" -Class "IIsWebServerSetting" -Filter "ServerComment = '$websiteName'"
}

Function getVirtualDirectory($website, $virtualDirectoryPath) {
    $virtualDirectory = New-Object System.DirectoryServices.DirectoryEntry("IIS://localhost/" + $website.Name + "/Root/$virtualDirectoryPath")
    if($virtualDirectory.Path) {
        Return $virtualDirectory
    } else {
        Return $null
    }
}

Function isIIS32BitMode() {
    $appPools= New-Object System.DirectoryServices.DirectoryEntry("IIS://localhost/W3SVC/AppPools")
    Return $appPools.Enable32BitAppOnWin64
}

Function validateDotNetPath($installationPath) {
    if (!(Test-Path $installationPath)) {
       Throw "Path [$installationPath] does not exist. Please enter a correct path for the .NET Framework installation directory"
    }

    $path = $installationPath.ToLower()
    $dotNetMode = @{$true=64;$false=32}[$path.Contains("framework64")]
    $iisMode = @{$true=32;$false=64}[(isIIS32BitMode)]
    if($dotNetMode -ne $iisMode) {
        Throw "IIS ${iisMode}bit is not compatible with .NET Framework ${dotNetMode}bit. Both must be either 32bit or 64bit. Please update the .NET installation path or the Enable32BitAppOnWin64 mode on all application pools to match."
    }
}

Function getDotNetVersions($installationPath) {
    [string[]] $versions = Get-ChildItem $installationPath | ?{ $_.PSIsContainer -and !$_.Name.StartsWith("v1") } | Select -ExpandProperty Name
    Return $versions
}

Function getBestMatchForDotNetVersion($installationPath, $partialVersion) {
    $partialVersion = $partialVersion.toLower()
    $versions = getDotNetVersions $installationPath
    foreach ($version in $versions) {
        if ($version.toLower().startsWith($partialVersion.toLower())) {
            Return $version
        }
    }

    Throw "Failed to find a .NET version matching '$partialVersion'. Available versions are [$($versions -join ',')]"
}

Function updateDotNetVersionForVirtualDirectory($vdPath, $dotNetInstallationPath, $version) {
    validateDotNetPath $dotNetInstallationPath
    $version = getBestMatchForDotNetVersion $dotNetInstallationPath $version
    & $dotNetInstallationPath\$version\aspnet_regiis.exe -sn $vdPath
    if(!$?) {
        Exit 1
    }
}

Function getDotNetVersionForVirtualDirectories($installationPath) {
    validateDotNetPath $installationPath
    $version = ([string[]](getDotNetVersions $installationPath))[0]
    Write-Host "Using .NET Version $version to list virtual directories and their .NET version"
    $pathsAndVersions = ([string[]](& $installationPath\$version\aspnet_regiis.exe -lk))
    $map = @{}
    foreach($pathAndVersion in $pathsAndVersions) {
        $splitPathAndVersion = $pathAndVersion.split()
        $vdPath = $splitPathAndVersion[0].trim().toLower()
        $vdDotNetVersion = $splitPathAndVersion[1].trim().toLower()
        $vdDotNetVersion = $vdDotNetVersion.substring(0,$vdDotNetVersion.lastIndexOf("."))
        $map[$vdPath] = "v$vdDotNetVersion"
    }
    Return $map
 }

