#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

Function Set-Logging-Properties ($websiteName)
{
    set-webconfigurationproperty "/system.applicationHost/sites/site[@Name='$($websiteName)']/logFile" -name enabled -value $deployed.logFileEnabled
    set-webconfigurationproperty "/system.applicationHost/sites/site[@Name='$($websiteName)']/logFile" -name logFormat -value $deployed.logFileLogFormat
    set-webconfigurationproperty "/system.applicationHost/sites/site[@Name='$($websiteName)']/logFile" -name logExtFileFlags -value $deployed.logFileLogExtFileFlags
    set-webconfigurationproperty "/system.applicationHost/sites/site[@Name='$($websiteName)']/logFile" -name directory -value $deployed.logFileDirectory
    set-webconfigurationproperty "/system.applicationHost/sites/site[@Name='$($websiteName)']/logFile" -name period -value $deployed.logFilePeriod
    if ($deployed.logFilePeriod -eq "MaxSize") {
        set-webconfigurationproperty "/system.applicationHost/sites/site[@Name='$($websiteName)']/logFile" -name truncateSize -value $deployed.logFileTruncateSize
    }
    set-webconfigurationproperty "/system.applicationHost/sites/site[@Name='$($websiteName)']/logFile" -name localTimeRollover -value $deployed.logFileLocalTimeRollover
}

Function Set-Limits-Property ($websiteName)
{
    set-webconfigurationproperty "/system.applicationHost/sites/site[@Name='$($websiteName)']/limits" -name connectionTimeout -value (New-TimeSpan -sec $deployed.limitsConnectionTimeout)
}

Function Set-Website-Id($websitePath, $websiteId)
{
    if($websiteId) {
        Write-Host "Changing website id to [$websiteId]."
        Set-ItemProperty -Path $websitePath -Name id -Value $websiteId
    }
}

Function Remove-Isapi-Locks()
{
    $assembly = [System.Reflection.Assembly]::LoadFrom("$env:systemroot\system32\inetsrv\Microsoft.Web.Administration.dll")

    # helper function to unlock sectiongroups
    function unlockSectionGroup($group)
    {
        foreach ($subGroup in $group.SectionGroups)
        {
            unlockSectionGroup($subGroup)
        }
        foreach ($section in $group.Sections)
        {
            $sectionName = $section.Name
            if($sectionName -eq 'isapiFilters'){
                Write-Host "Unlocking [$sectionName]"
                $section.OverrideModeDefault = "Allow"
                Write-Host "Unlocked [$sectionName]"
            }

        }
    }

    # initial work
    # load ServerManager
    $mgr = new-object Microsoft.Web.Administration.ServerManager
    # load appHost config
    $conf = $mgr.GetApplicationHostConfiguration()
    $sec = $conf.GetSection('system.webServer/isapiFilters')
    $sec.OverrideMode = "Allow"
    # unlock all sections in system.webServer
    unlockSectionGroup(
    $conf.RootSectionGroup.SectionGroups['system.webServer'])
    $mgr.CommitChanges()

}

Function Set-Isapi-Filters($websitePath)
{
    Remove-Isapi-Filters($websitePath)
    Remove-Isapi-Locks
    # Add deployed ISAPI filters
    foreach ($isapiFilter in $deployed.isapiFilters) {
        Write-Host "Adding ISAPI filter [$($isapiFilter.filterName)]."
        Write-host $isapiFilter.filterName $isapiFilter.executable $websitePath
        Add-WebConfiguration -Filter /system.webServer/isapiFilters -PSPath $websitePath -Value @{name=$isapiFilter.filterName;path=$isapiFilter.executable}
    }
}

Function Remove-Isapi-Filters($websitePath)
{
    # Revert to parent
    # Delete all ISAPI filters
    $currentFilters = Get-WebIsapiFilters $websitePath
    $parentFilters = Get-WebIsapiFilters "iis:\sites"
    if ($parentFilters -ne $Null) {
        Compare-Object $currentFilters $parentFilters | ForEach-Object {
            $filter = $_.InputObject
            $fid = "$id/$($filter.name)"
            Clear-WebItem $filter
        }
    } ElseIf ($currentFilters -ne $Null) {
         Foreach ($filter in $currentFilters) {
             $fid = "$($filter.name)"
             Clear-WebItem $filter
        }
    }
}


Function Get-WebIsapiFilters($websitePath)
{
    Get-WebConfiguration -PSPath $websitePath -Filter "/system.webServer/isapiFilters/filter"
}

Function Clear-WebItem($isapiFilter)
{
    Write-Host "Clearing ISAPI filter [$($isapiFilter.name)]."
    Clear-WebConfiguration -PSPath $($isapiFilter.PSPath) -Filter $($isapiFilter.ItemXPath) -Location $($isapiFilter.Location)
}
# Get provided website name or use the deployed name.
$websiteName = if($deployed.websiteName) { $deployed.websiteName } else { $deployed.name }

# Verify website name is not being changed.
if($previousDeployed) {
    $previousWebsiteName = if($previousDeployed.websiteName) { $previousDeployed.websiteName } else { $previousDeployed.name }
    if($websiteName -ne $previousWebsiteName) {
        Write-Host "Renaming a website is not supported. Undeploy and deploy the website instead."
        Exit 1
    }
}

# Verify the website has at least one binding specified, otherwise it won't be able to start later
if(!$deployed.bindings){
    Write-Host "Website must have at least one binding specified."
    Exit 1
}

# check if https bindings contain certificate
foreach ($binding in $deployed.bindings) {
    if( "$($binding.protocol)" -eq "https" ) {
       if (!$($binding.certificateName)) {
           Write-Host "When an https binding is used, a certificatename must be specified."
           Exit 1
       }
    }
}

# Create physical path if it doesn't exist yet.
if (!(Test-Path $deployed.physicalPath)) {
    Write-Host "Creating empty website physical path [$($deployed.physicalPath)]."
    if(!(New-Item $deployed.physicalPath -Type Directory -Force)) {
        Write-Host "Cannot create directory [$($deployed.physicalPath)]."
        Exit 1
    }
}

# Check whether website already exists and then either modify or create it.
$websitePath = "IIS:\Sites\$websiteName"
if (Test-Path $websitePath) {
    Write-Host "Modifying existing website [$websiteName]."
 
    Set-ItemProperty -Path $websitePath -Name PhysicalPath -Value $deployed.physicalPath
    Set-Website-Id $websitePath $($deployed.websiteId)
    Set-Logging-Properties $websiteName
    Set-Limits-Property $websiteName
} else {
    Write-Host "Creating new website [$websiteName]."

    $params = @{ Name=$websiteName; PhysicalPath=$deployed.physicalPath }

    if (!(Get-Website)) {
        Write-Host "INFO: Activating workaround for PowerShell bug that occurs when there are no websites yet."
        $params.id = 1
    }

    New-Website @params | Out-Null
    Set-Website-Id $websitePath $($deployed.websiteId)
    Set-Logging-Properties $websiteName
    Set-Limits-Property $websiteName
}

# Set application pool.
if($deployed.applicationPoolName) {
    Set-ItemProperty -Path $websitePath -Name ApplicationPool -Value $deployed.applicationPoolName
}

# Remove existing bindings.
foreach ($existingBinding in (Get-WebBinding -Name $websiteName)) {
    Write-Host "Removing existing binding [$($existingBinding.bindingInformation)] from website [$websiteName]."

    $bindingProtocol = $($existingBinding.protocol)

    if( "$bindingProtocol" -eq "https" )
    {
        $bindingAdress = $($existingBinding.bindingInformation).split(':')[0]
        $bindingPort = $($existingBinding.bindingInformation).split(':')[1]
        if( "$bindingAdress" -eq "*" )
        {
            $bindingAdress = "0.0.0.0"
        }
        Write-Host "Removing related SslBinding [$($existingBinding.bindingInformation)] from website [$websiteName]"
        remove-item -path IIS:\SslBindings\$bindingAdress!$bindingPort
    }

    Remove-WebBinding -InputObject $existingBinding
}

# Add bindings.
foreach ($binding in $deployed.bindings) {
    Write-Host "Adding binding [$($binding.name)] for website [$websiteName]."

    $params = @{ Name=$websiteName; Protocol=$binding.protocol; Port=$binding.port; IPAddress=$binding.ipAddress }

    if ($binding.hostHeader) {
        $params.hostHeader = $binding.hostHeader
    }

    if( "$($binding.protocol)" -eq "https" ) {
        if( "$($binding.enableSni)" -eq "True" ) {
            
            Write-Host "Found Require Server Name Indication set to [$($binding.enableSni)]"
            $params.sslFlags = "1"
        }
        if( "$($binding.sslRequired)" -eq "True" ) {

            Write-Host "Found SSL Require for content of a website or application set to [$($binding.sslRequired)]"
            # set SSL Required for content of a website or application

            $useClientCert = switch ( $($binding.certRequired) )
            {
                'Ignore' { 'Ssl'    }
                'Accept' { 'Ssl,SslNegotiateCert'    }
                'Require' { 'Ssl,SslNegotiateCert,SslRequireCert'   }
            }
            Write-Host "Found SSL Require for content of a website or application set to [$useClientCert]"
            Set-WebConfiguration -PSPath IIS:\ -Location $location -filter "system.webServer/security/access" -value $useClientCert

        }
    }

    New-WebBinding @params | Out-Null

   if( "$($binding.protocol)" -eq "https" ) {

        $bindingPort = $($binding.port)
        $bindingAdress = $($binding.ipAddress)
        if( "$bindingAdress" -eq "*" )
        {
            $bindingAdress = "0.0.0.0"
        }

        $isSslBindingExists = Test-Path IIS:\SslBindings\$bindingAdress!$bindingPort
        if( $isSslBindingExists )
        {
            Write-Host "Removing existing SslBinding [$($binding.name)] for website [$websiteName]."
            remove-item -path IIS:\SslBindings\$bindingAdress!$bindingPort
        }

        Write-Host "Adding related SslBinding [$($binding.name)] for website [$websiteName]."
        get-item cert:\LocalMachine\My\$($binding.certificateName) | new-item -path IIS:\SslBindings\$bindingAdress!$bindingPort
    }
}

Set-Isapi-Filters $websitePath
setAuthentication $deployed $websiteName
setDirectoryBrowsing $deployed $websiteName

