#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

# Get provided application pool name or use the deployed name.
$applicationPoolName = if ($deployed.applicationPoolName)
{
    $deployed.applicationPoolName
}
else
{
    $deployed.name
}
$cleanRecycleTimes = $deployed.cleanRecycleTimes -as [bool]
$printLog = $true

#returns an array with the value of 'specic times' in iis
function Get-Current-SpecificTime-Data
{
    $itemsInIis = (Get-ItemProperty $appPoolPath -Name Recycling.periodicRestart.schedule.collection).value -F 'HHmmsss'
    $arr = @($itemsInIis.split(' '))
    return $arr
}


function Is-Valid-Time
{
    param($time, $itemsInIis)
    if ( $itemsInIis.Contains($time))
    {
        return $false
    }
    return $true
}

# Verify application pool name is not being changed.
if ($previousDeployed)
{
    $previousApplicationPoolName = if ($previousDeployed.applicationPoolName)
    {
        $previousDeployed.applicationPoolName
    }
    else
    {
        $previousDeployed.name
    }
    if ($applicationPoolName -ne $previousApplicationPoolName)
    {
        Write-Host "Renaming an application pool is not supported. Undeploy and deploy the application pool instead."
        Exit 1
    }
}

# Verify username and password are provided if serviceAccount if set to 'SpecificUser'.
if ($deployed.serviceAccount -eq "SpecificUser")
{
    if (!$deployed.username)
    {
        throw "Username is required when service account is [SpecificUser]."
    }
    if (!$deployed.password)
    {
        throw "Password is required when service account is [SpecificUser]."
    }
}

# Check whether application pool already exists and then either retrieve or create it.
$appPoolPath = "IIS:\AppPools\$applicationPoolName"
if (Test-Path $appPoolPath)
{
    Write-Host "Modifying existing application pool [$applicationPoolName]."
    $applicationPool = Get-Item $appPoolPath
}
else
{
    Write-Host "Creating new application pool [$applicationPoolName]."

    $applicationPool = New-Item $appPoolPath

    # Sleep after creating the application pool, IIS is sometimes not ready to stop it immediately
    Write-Host "Waiting for application pool [$applicationPoolName] to initialize."
    Start-Sleep -Seconds $deployed.sleepAfterCreate

    # Stopping application pool so that we can configure it correctly before starting it properly from start-application-pool.ps1
    Stop-WebAppPool $applicationPoolName

    #when a pool gets created iis adds 04:00:00 as a default time. Setting this to true on creation, will allow it to be cleared in the 'if check' bellow so it can start from fresh.
    $cleanRecycleTimes = $true
    #used to not print out removing the default time at first time
    $printLog = $false
}

# Set application pool properties.

if ($deployed.managedRuntimeVersion -eq "nmc")
{
    $applicationPool.managedRuntimeVersion = ''
}
Else
{
    $applicationPool.managedRuntimeVersion = $deployed.managedRuntimeVersion
}

$applicationPool.managedPipelineMode = if ($deployed.useClassicManagedPipeline)
{
    'Classic'
}
else
{
    'Integrated'
}
$applicationPool.enable32BitAppOnWin64 = $deployed.enable32BitAppOnWin64
$applicationPool.queueLength = $deployed.queueLength -as [int]

# Process Model settings
$applicationPool.ProcessModel.IdentityType = $deployed.serviceAccount
if ($deployed.serviceAccount -eq "SpecificUser")
{
    $applicationPool.ProcessModel.UserName = $deployed.username
    $applicationPool.ProcessModel.Password = $deployed.password
}
$applicationPool.ProcessModel.PingInterval = (New-TimeSpan -sec $deployed.processModelPingInterval).toString()
$applicationPool.ProcessModel.PingResponseTime = (New-TimeSpan -sec $deployed.processModelPingResponseTime).toString()
$applicationPool.ProcessModel.StartupTimeLimit = (New-TimeSpan -sec $deployed.processModelStartupTimeLimit).toString()
$applicationPool.ProcessModel.ShutdownTimeLimit = (New-TimeSpan -sec $deployed.processModelShutdownTimeLimit).toString()
$applicationPool.ProcessModel.MaxProcesses = $deployed.processModelMaxProcesses -as [int]
$applicationPool.ProcessModel.IdleTimeout = (New-TimeSpan -min $deployed.processModelIdleTimeout).toString()
$applicationPool.ProcessModel.pingingEnabled = $deployed.processModelPingEnabled -as [bool]
# Failure settings
$applicationPool.Failure.RapidFailProtection = $deployed.failureRapidFailProtection -as [bool]
$applicationPool.Failure.RapidFailProtectionInterval = (New-TimeSpan -min $deployed.failureRapidFailProtectionInterval).toString()
$applicationPool.Failure.RapidFailProtectionMaxCrashes = $deployed.failureRapidFailProtectionMaxCrashes -as [int]

# Recycling settings
$applicationPool.Recycling.PeriodicRestart.Time = (New-TimeSpan -min $deployed.recyclingPeriodicRestartTime).toString()

# Save application pool.
$applicationPool | Set-Item


if ($cleanRecycleTimes)
{
    $itemsInIis = Get-Current-SpecificTime-Data
    Remove-ItemProperty $appPoolPath -Name Recycling.periodicRestart.schedule.collection #clear values
    if ($printLog)
    {
        Write-Host "Removed the following times from IIS: $itemsInIis"
    }

}

#Periodic restart setting
$deployed.recyclingPeriodicRestartSchedule | foreach {
    #refresh every time something gets added
    $itemsInIis = Get-Current-SpecificTime-Data

    Write-Host "Current items in iis: [$itemsInIis]"

    $arr = $_ -split ':'

    if ($arr.Count -eq "2")
    {
        $_ = $_ + ":00"
    }

    $time = [TimeSpan]::Parse($_).toString()
    if (Is-Valid-Time $time $itemsInIis)
    {
        Write-Host "Adding $_ to specific times."
        New-ItemProperty $appPoolPath -Name Recycling.periodicRestart.schedule -Value @{ value = ($time) }
    }
}

$itemsInIis = Get-Current-SpecificTime-Data
Write-Host "Final recycle 'Specic Times' from IIS: [$itemsInIis]"

