package com.xebialabs.deployit.plugin.glassfish.step;

import java.util.List;
import java.util.Map;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.glassfish.container.CliManagedContainer;

import static com.google.common.base.Preconditions.checkNotNull;

abstract public class StepFactory {

    protected List<String> libraries;

    private StepFactory() {}

    public StepFactory withLibraries(List<String> libraries) {
        this.libraries = Lists.newArrayList(libraries);
        return this;
    }

    public BaseStep createStep(final String script, final int order, final Map<String, Object> pyCtx, String description, CliManagedContainer container) {
        BaseStep step = newStep(script, order, pyCtx, description, container);
        if(libraries != null){
            step.setAdditionalLibraries(libraries);
        }
        return step;
    }

    abstract protected BaseStep newStep(final String script, final int order, final Map<String, Object> pyCtx, String description, CliManagedContainer container);

    public static StepFactory stagingStepFactory(final Artifact artifact){
        return new StagingStepFactory(artifact);
    }

    public static StepFactory deploymentStepFactory(){
        return new DeploymentStepFactory();
    }

    private static class StagingStepFactory extends StepFactory {

        private final Artifact artifact;

        private StagingStepFactory(final Artifact artifact) {
            checkNotNull(artifact != null, "Artifact must be specified");
            this.artifact = artifact;
        }

        @Override
        protected BaseStep newStep(final String script, final int order, final Map<String, Object> pyCtx, final String description, final CliManagedContainer container) {
            return new CliStageableStep(artifact, script, order, pyCtx, description, container);
        }
    }

    private static class DeploymentStepFactory extends StepFactory {

        @Override
        protected BaseStep newStep(final String script, final int order, final Map<String, Object> pyCtx, final String description, final CliManagedContainer container) {
            return new CliDeploymentStep(script, order, pyCtx, description, container);
        }
    }
}
