package com.xebialabs.deployit.plugin.glassfish.session;

import java.io.*;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Strings;
import com.google.common.io.ByteStreams;
import com.google.common.util.concurrent.SimpleTimeLimiter;
import com.google.common.util.concurrent.UncheckedTimeoutException;

import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.overthere.*;
import com.xebialabs.overthere.util.OverthereUtils;

import static com.google.common.collect.Lists.newArrayList;

/*
  TODO:
         This class is not integrated into the CliDaemon, because we are not getting back the
         prompt from asadmin via the inputstream.  Need to investigate that and this class will be
         good to go.
 */

public class InteractiveSession implements CliSession {

    private int idleResponseTimeInSeconds = 60;

    private OverthereProcess process;
    private OverthereConnection connection;
    private Pattern promptPattern;
    private Host host;
    private String driverScriptContent;
    private String driverScriptName;
    private SimpleTimeLimiter limiter = new SimpleTimeLimiter();
    private CliSession timeoutEnabledSession;

    public InteractiveSession(Host host, String driverScriptContent, Pattern promptPattern) {
        this.host = host;
        this.driverScriptContent = driverScriptContent;
        this.promptPattern = promptPattern;
    }

    public boolean isConnected() {
        return connection != null && process != null;
    }


    public void disconnect() {
        if (process != null) {
            process.destroy();
            process = null;
        }

        if (connection != null) {
            connection.close();
            connection = null;
        }
    }

    public void connect() {
        connect(true);

    }
    private void connect(boolean enableTimeout) {
        disconnect();
        connection = host.getConnection();
        process = connection.startProcess(setupCmdLine());
        final StringBuilder outputBuffer = new StringBuilder();

        if (!enableTimeout) {
            waitForSessionToStart(outputBuffer);
            return;
        }

        try {
            limiter.callWithTimeout(new Callable<Object>() {
                @Override
                public Object call() throws Exception {
                    waitForSessionToStart(outputBuffer);
                    return null;
                }
            }, idleResponseTimeInSeconds, TimeUnit.SECONDS, true);
        } catch (UncheckedTimeoutException timeout) {
            throw new RuntimeIOException("Failed to start interactive session due to time out. Output:\n " + outputBuffer);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private CmdLine setupCmdLine() {
        OverthereFile driverFile = uploadDriver();
        CmdLine cmdLine = new CmdLine();
        cmdLine.addArgument(driverFile.getPath());
        return cmdLine;
    }

    private OverthereFile uploadDriver() {
        OverthereFile driverFile;
        if (Strings.isNullOrEmpty(driverScriptName)) {
            driverFile = connection.getTempFile(driverScriptName);
        } else {
            driverFile = connection.getTempFile("glassfish","driver");
        }

        OverthereUtils.write(driverScriptContent.getBytes(),driverFile);
        driverFile.setExecutable(true);
        return driverFile;
    }

    public Response execute(final String command) {
        final List<String> outputBuffer = newArrayList();
        try {
                limiter.callWithTimeout(new Callable<String>() {
                    @Override
                    public String call() throws Exception {
                        doExecute(command, outputBuffer);
                        return null;
                    }
                }, idleResponseTimeInSeconds, TimeUnit.SECONDS, true);
        } catch (UncheckedTimeoutException timeout) {
            return new Response(1,outputBuffer, "Failed to execute command [" + command + "] due to time out.");
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        return new Response(0,outputBuffer);
    }

    private void doExecute(String command, List<String> outputBuffer) {
        if (!isConnected()) {
            connect(false);
        }

        writeCommandToStandardInput(command);
        BufferedReader inputStream = new BufferedReader(new InputStreamReader(process.getStdout()));

        while (true) {
            String line = null;
            try {
                line = inputStream.readLine();
            } catch (UncheckedTimeoutException timeout) {
                throw new RuntimeIOException("Failed to execute command [" + command + "] due to time out. Output:\n " + outputBuffer);
            } catch (IOException exc) {
                throw new RuntimeIOException("Failed to execute command [" + command + "] due to time out. Output:\n " + outputBuffer, exc);
            }

            if (line == null) {
                break;
            }

            Matcher matcher = promptPattern.matcher(line);

            if (matcher.matches()) {
                break;
            }
            outputBuffer.add(line);
        }
    }

    private void writeCommandToStandardInput(String command) {
        String finalCommand;
        if (command.charAt(command.length() - 1) == '\n') {
            finalCommand = command;
        } else {
            finalCommand = command + "\n";
        }

        OutputStream stdin = process.getStdin();
        try {
            stdin.write(finalCommand.getBytes());
            stdin.flush();
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    private void waitForSessionToStart(StringBuilder outputBuffer) {
        InputStream inputStream = process.getStdout();
        while (true) {
            int cInt = 0;
            try {
                cInt = inputStream.read();
            } catch (IOException exc) {
                throw new RuntimeIOException("Failed to start interactive session", exc);
            }

            if (cInt == -1) {
                exitBecauseProcessFailedToStart(process.getStderr());
            }

            outputBuffer.append((char) cInt);
            //TODO:  Check why prompt ("asadmin>") is not on the inputstream
            Matcher matcher = promptPattern.matcher(outputBuffer);
            if (matcher.matches()) {
                break;
            }
        }
    }

    private void exitBecauseProcessFailedToStart(final InputStream stderr) {
        int exitCode = -1;
        String errMsg = "";

        try {
            exitCode = process.waitFor();
            if (stderr.available() > 0) {
                errMsg = new String(ByteStreams.toByteArray(stderr));
            }
        } catch (InterruptedException exc) {
            logger.error("Interrupted while waiting for " + process + " to complete");
            Thread.currentThread().interrupt();
        } catch (IOException e) {
            logger.error("Failed to read error stream.", e);
        }
        throw new RuntimeIOException("Failed to start interactive session with exit code " + exitCode + ". " + errMsg);
    }

    public Pattern getPromptPattern() {
        return promptPattern;
    }

    public int getIdleResponseTimeInSeconds() {
        return idleResponseTimeInSeconds;
    }

    public void setIdleResponseTimeInSeconds(final int idleResponseTimeInSeconds) {
        this.idleResponseTimeInSeconds = idleResponseTimeInSeconds;
        this.timeoutEnabledSession = null;
    }

    public String getDriverScriptName() {
        return driverScriptName;
    }

    public void setDriverScriptName(final String driverScriptName) {
        this.driverScriptName = driverScriptName;
    }

    private static final Logger logger = LoggerFactory.getLogger(InteractiveSession.class);

}
