package com.xebialabs.deployit.plugin.glassfish.deployed;

import java.util.HashMap;
import java.util.List;
import com.google.common.base.Strings;

import com.xebialabs.deployit.plugin.api.deployment.planning.*;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.inspection.Inspect;
import com.xebialabs.deployit.plugin.api.inspection.InspectionContext;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployed;
import com.xebialabs.deployit.plugin.glassfish.container.CliManagedContainer;
import com.xebialabs.deployit.plugin.glassfish.step.BaseStep;
import com.xebialabs.deployit.plugin.glassfish.step.CliInspectionStep;
import com.xebialabs.deployit.plugin.glassfish.constants.ContainerRestartStrategy;
import com.xebialabs.deployit.plugin.glassfish.constants.DeployedRestartStrategy;
import com.xebialabs.deployit.plugin.glassfish.step.StepFactory;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.plugin.glassfish.step.StepFactory.deploymentStepFactory;

@Metadata(virtual = true, description = "Base for all deployed that utilize the Glassfish Cli for configuration")
public class CliManagedDeployed<D extends Deployable, C extends CliManagedContainer> extends BaseDeployed<D, C> {

    @Property(hidden = true, defaultValue = "60", description = "The order of the step in the step list for the create operation.")
    private int createOrder;

    @Property(hidden = true, defaultValue = "40", description = "The order of the step in the step list for the destroy operation.")
    private int destroyOrder;

    @Property(hidden = true, defaultValue = "60", description = "The order of the step in the step list for the modify operation.")
    private int modifyOrder;

    @Property(hidden = true, defaultValue = "50", description = "The order of the step in the step list for the noop operation.")
    private int noopOrder;

    @Property(hidden = true, defaultValue = "Create")
    private String createVerb;

    @Property(hidden = true, defaultValue = "Modify")
    private String modifyVerb;

    @Property(hidden = true, defaultValue = "Destroy")
    private String destroyVerb;

    @Property(hidden = true, defaultValue = "Modify")
    private String noopVerb;

    @Property(required = true, hidden = true, description = "Classpath to the script that is uploaded and executed on the generic container for the create operation.")
    private String createScript;

    @Property(required = false, hidden = true, description = "Classpath to the script that is uploaded and executed on the generic container for the modify operation.")
    private String modifyScript;

    @Property(required = false, hidden = true, description = "Classpath to the script that is uploaded and executed on the generic container for the destroy operation.")
    private String destroyScript;

    @Property(required = false, hidden = true, description = "Classpath to the script that is uploaded and executed on the generic container for the noop operation.")
    private String noopScript;

    @Property(required = false, hidden = true, description = "Classpath to the script used to inspect the generic container.")
    private String inspectScript;

    @Property(hidden = true, required = false, description = "List of python library scripts that should be automatically loaded when using a Glassfish CLI script.")
    private List<String> libraryScripts = newArrayList();

    @Property(hidden = true, required = false, description = "Python script invoked to start a Java EE artifact or Java EE resource")
    private String startScript;

    @Property(hidden = true, defaultValue = "Start", description = "The word that is used to prefix a step description for the start operation.")
    private String startVerb;

    @Property(hidden = true, defaultValue = "90", description = "The order in which a start step will be executed.")
    private int startOrder;

    @Property(hidden = true, required = false, description = "Python script invoked to stop a Java EE artifact or Java EE resource")
    private String stopScript;

    @Property(hidden = true, defaultValue = "Stop", description = "The word that is used to prefix a step description for the stop operation.")
    private String stopVerb;

    @Property(hidden = true, defaultValue = "10", description = "The order in which a stop step will be executed.")
    private int stopOrder;

    @Property(defaultValue = "NONE", hidden=true, category = "Restart Strategy", description = "The logical stop/start sequence for the container affected by the deployed. " +
            "STOP_START : stop container, undeploy, deploy, start container. RESTART : undeploy, deploy, stop container, start container.")
    private ContainerRestartStrategy containerRestartStrategy = ContainerRestartStrategy.NONE;

    @Property(defaultValue = "NONE", hidden=true, category = "Restart Strategy", description = "The logical stop/start sequence for the deployed. " +
            "STOP_START : stop, undeploy, deploy, start. STOP: stop, undeploy, deploy. START: undeploy, deploy, start.")
    private DeployedRestartStrategy restartStrategy = DeployedRestartStrategy.STOP_START;

    @Property(defaultValue = "false", hidden=true, category = "Restart Strategy", description = "When true, the restart strategy is apply for a NOOP operation")
    private boolean applyRestartStrategyOnNoop;

    protected boolean addStep(final StepFactory stepFactory, DeploymentPlanningContext ctx, String script, int order, String verb, Delta delta, boolean checkpoint) {
        if (!Strings.isNullOrEmpty(script)) {
            HashMap<String, Object> pyCtx = createPythonVars();
            pyCtx.put("delta", delta);
            pyCtx.put("deployedApplication", ctx.getDeployedApplication());
            BaseStep step = stepFactory.withLibraries(libraryScripts).createStep(script, order, pyCtx, getDescription(verb), this.getContainer());
            if (checkpoint) {
                ctx.addStepWithCheckpoint(step, delta);
            } else {
                ctx.addStep(step);
            }
            return true;
        }

        return false;
    }

    protected HashMap<String, Object> createPythonVars() {
        HashMap<String, Object> pyCtx = newHashMap();
        pyCtx.put("deployed", this);
        pyCtx.put("container", this.getContainer());
        return pyCtx;
    }

    public String getDescription(String verb) {
        return String.format("%s %s on %s", verb, getName(), getContainer().getName());
    }

    @Inspect
    public void inspect(InspectionContext ctx) {
        if (!Strings.nullToEmpty(getInspectScript()).trim().isEmpty()) {
            String desc = "Inspect " + getName() + " in container " + getContainer().getId();
            CliInspectionStep step = new CliInspectionStep(getInspectScript(), 0, createPythonVars(), desc, getContainer());
            step.setAdditionalLibraries(libraryScripts);
            ctx.addStep(step);
        }
    }

    @Create
    public void executeCreate(DeploymentPlanningContext ctx, Delta delta) {
        executeCreateWithStepFactory(deploymentStepFactory(), ctx, delta);
    }

    protected void executeCreateWithStepFactory(final StepFactory stepFactory, final DeploymentPlanningContext ctx, final Delta delta) {
        @SuppressWarnings("unchecked") CliManagedDeployed<D, C> deployed = (CliManagedDeployed<D, C>) delta.getDeployed();
        deployed.addStep(stepFactory, ctx, getCreateScript(), getCreateOrder(), getCreateVerb(), delta, true);
        addStartStep(ctx);
    }

    private void addStartStep(DeploymentPlanningContext ctx) {
        if (restartStrategy == DeployedRestartStrategy.STOP_START || restartStrategy == DeployedRestartStrategy.START) {
            addStep(deploymentStepFactory(), ctx, getStartScript(), getStartOrder(), getStartVerb(), null, false);
        }
    }

    private void addStopStep(DeploymentPlanningContext ctx) {
        if (restartStrategy == DeployedRestartStrategy.STOP_START || restartStrategy == DeployedRestartStrategy.STOP) {
            addStep(deploymentStepFactory(), ctx, getStopScript(), getStopOrder(), getStopVerb(), null, false);
        }
    }

    @Destroy
    @SuppressWarnings("unchecked")
    public void executeDestroy(DeploymentPlanningContext ctx, Delta delta) {
        CliManagedDeployed<D, C> deployed = (CliManagedDeployed<D, C>) delta.getPrevious();
        deployed.addStep(deploymentStepFactory(), ctx, getDestroyScript(), getDestroyOrder(), getDestroyVerb(), delta, true);
        deployed.addStopStep(ctx);
    }

    @Noop
    @SuppressWarnings("unchecked")
    public void executeNoop(DeploymentPlanningContext ctx, Delta delta) {
        CliManagedDeployed<D, C> deployed = (CliManagedDeployed<D, C>) delta.getDeployed();
        deployed.addStep(deploymentStepFactory(), ctx, getNoopScript(), getNoopOrder(), getNoopVerb(), delta, false);
        if (applyRestartStrategyOnNoop) {
            addStartStep(ctx);
            addStopStep(ctx);
        }
    }

    @Modify
    @SuppressWarnings("unchecked")
    public void executeModify(DeploymentPlanningContext ctx, Delta delta) {
        CliManagedDeployed<D, C> deployed = (CliManagedDeployed<D, C>) delta.getDeployed();
        boolean added = deployed.addStep(deploymentStepFactory(), ctx, getModifyScript(), getModifyOrder(), getModifyVerb(), delta, true);
        if (added) {
            addStartStep(ctx);
            addStopStep(ctx);
        } else {
            CliManagedDeployed<D, C> previousDeployed = (CliManagedDeployed<D, C>) delta.getPrevious();
            // Apply the most current start/stop strategy the user has selected.
            previousDeployed.restartStrategy = deployed.restartStrategy;
            previousDeployed.containerRestartStrategy = deployed.containerRestartStrategy;

            previousDeployed.executeDestroy(ctx, delta);
            deployed.executeCreate(ctx, delta);
        }

    }

    public List<String> getLibraryScripts() {
        return libraryScripts;
    }

    public void setLibraryScripts(List<String> libraryScripts) {
        this.libraryScripts = libraryScripts;
    }

    public String getCreateScript() {
        return createScript;
    }

    public void setCreateScript(String createScript) {
        this.createScript = createScript;
    }

    public String getModifyScript() {
        return modifyScript;
    }

    public void setModifyScript(String modifyScript) {
        this.modifyScript = modifyScript;
    }

    public String getDestroyScript() {
        return destroyScript;
    }

    public void setDestroyScript(String destroyScript) {
        this.destroyScript = destroyScript;
    }

    public String getNoopScript() {
        return noopScript;
    }

    public void setNoopScript(String noopScript) {
        this.noopScript = noopScript;
    }

    public int getCreateOrder() {
        return createOrder;
    }

    public void setCreateOrder(int createOrder) {
        this.createOrder = createOrder;
    }

    public int getDestroyOrder() {
        return destroyOrder;
    }

    public void setDestroyOrder(int destroyOrder) {
        this.destroyOrder = destroyOrder;
    }

    public int getModifyOrder() {
        return modifyOrder;
    }

    public void setModifyOrder(int modifyOrder) {
        this.modifyOrder = modifyOrder;
    }

    public int getNoopOrder() {
        return noopOrder;
    }

    public void setNoopOrder(int noopOrder) {
        this.noopOrder = noopOrder;
    }

    public String getCreateVerb() {
        return createVerb;
    }

    public void setCreateVerb(String createVerb) {
        this.createVerb = createVerb;
    }

    public String getModifyVerb() {
        return modifyVerb;
    }

    public void setModifyVerb(String modifyVerb) {
        this.modifyVerb = modifyVerb;
    }

    public String getDestroyVerb() {
        return destroyVerb;
    }

    public void setDestroyVerb(String destroyVerb) {
        this.destroyVerb = destroyVerb;
    }

    public String getNoopVerb() {
        return noopVerb;
    }

    public void setNoopVerb(String noopVerb) {
        this.noopVerb = noopVerb;
    }

    public String getInspectScript() {
        return inspectScript;
    }

    public void setInspectScript(String inspectScript) {
        this.inspectScript = inspectScript;
    }

    public String getStartScript() {
        return startScript;
    }

    public void setStartScript(final String startScript) {
        this.startScript = startScript;
    }

    public String getStartVerb() {
        return startVerb;
    }

    public void setStartVerb(final String startVerb) {
        this.startVerb = startVerb;
    }

    public int getStartOrder() {
        return startOrder;
    }

    public void setStartOrder(final int startOrder) {
        this.startOrder = startOrder;
    }

    public String getStopScript() {
        return stopScript;
    }

    public void setStopScript(final String stopScript) {
        this.stopScript = stopScript;
    }

    public String getStopVerb() {
        return stopVerb;
    }

    public void setStopVerb(final String stopVerb) {
        this.stopVerb = stopVerb;
    }

    public int getStopOrder() {
        return stopOrder;
    }

    public void setStopOrder(final int stopOrder) {
        this.stopOrder = stopOrder;
    }

    public ContainerRestartStrategy getContainerRestartStrategy() {
        return containerRestartStrategy;
    }

    public void setContainerRestartStrategy(final ContainerRestartStrategy containerRestartStrategy) {
        this.containerRestartStrategy = containerRestartStrategy;
    }

    public DeployedRestartStrategy getRestartStrategy() {
        return restartStrategy;
    }

    public void setRestartStrategy(final DeployedRestartStrategy restartStrategy) {
        this.restartStrategy = restartStrategy;
    }

    public boolean isApplyRestartStrategyOnNoop() {
        return applyRestartStrategyOnNoop;
    }

    public void setApplyRestartStrategyOnNoop(final boolean applyRestartStrategyOnNoop) {
        this.applyRestartStrategyOnNoop = applyRestartStrategyOnNoop;
    }
}
