package com.xebialabs.deployit.plugin.glassfish.session;

import java.io.IOException;
import java.io.Serializable;
import java.io.StringWriter;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.annotations.VisibleForTesting;

import com.xebialabs.deployit.plugin.api.execution.ExecutionContextListener;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.generic.freemarker.ArtifactUploader;
import com.xebialabs.deployit.plugin.generic.freemarker.CiAwareObjectWrapper;
import com.xebialabs.deployit.plugin.glassfish.container.CliBasedContainer;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.overthere.OperatingSystemFamily;
import com.xebialabs.overthere.RuntimeIOException;

import freemarker.template.Template;
import freemarker.template.TemplateException;

import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder.getConfiguration;

public class CliDaemon implements ExecutionContextListener, Serializable {

    private final Host connectingHost;

    private CliBasedContainer container;
    private String asadminCmdline;
    private String asadminFileName;

    private transient CliSession singleExecutionSession;

    public CliDaemon(CliBasedContainer container, Host connectingHost) {
        this.container = container;
        this.connectingHost = connectingHost;
        this.asadminFileName = (connectingHost.getOs() == OperatingSystemFamily.WINDOWS) ? "asadmin.bat" : "asadmin";
        loadAsadminCmdLine();
    }

    private void loadAsadminCmdLine() {
        try {
            Map<String,Object> vars = newHashMap();
            vars.put("container", container);
            Template template = getConfiguration().getTemplate("glassfish/cli/asadmin-cmdline-driver.ftl");
            StringWriter sw = new StringWriter();
            template.createProcessingEnvironment(vars, sw, new CiAwareObjectWrapper(NULL_FILE_UPLOADER, false)).process();
            sw.write(' ');
            asadminCmdline = sw.toString();
            if (logger.isTraceEnabled()) {
                StringWriter debug = new StringWriter();
                template.createProcessingEnvironment(vars, debug, new CiAwareObjectWrapper(NULL_FILE_UPLOADER, true)).process();
                debug.write(' ');
                logger.trace("ascommand drive content:\n" + debug.toString());
            }
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        } catch (TemplateException e) {
            throw new RuntimeException(e);
        }

    }

    public Response executeCliCommandWithoutDaemon(ExecutionContext context, String cmd) {
        return getSession().execute(asadminCmdline + " " + cmd);
    }

    public Response executeCliCommand(ExecutionContext context, String cmd) {
        //Hook point to use the InteractSession when fully coded and tested.
        //For now run with single session.
        return executeCliCommandWithoutDaemon(context, cmd);
    }

    public void disconnect() {
        if (singleExecutionSession != null) {
            singleExecutionSession.disconnect();
        }
    }

    public void connect() {
        getSession().connect();
    }

    @Override
    public void contextDestroyed() {
        disconnect();
    }

    private static final ArtifactUploader NULL_FILE_UPLOADER = new ArtifactUploader() {

        private final String EMPTY="";

        @Override
        public String upload(Artifact file) {
            return EMPTY;
        }
    };

    private CliSession getSession() {
        if (singleExecutionSession == null) {
            singleExecutionSession = new SingleExecutionSession(connectingHost);
        }

        return singleExecutionSession;
    }

    @VisibleForTesting
    String getAsadminCmdline() {
        return asadminCmdline;
    }

    private static final Logger logger = LoggerFactory.getLogger(CliDaemon.class);

}
