package com.xebialabs.deployit.plugin.generic.deployed;

import java.util.Map;

import com.xebialabs.deployit.plugin.api.deployment.planning.Create;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.planning.Destroy;
import com.xebialabs.deployit.plugin.api.deployment.planning.Modify;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.api.validation.Placeholders;
import com.xebialabs.deployit.plugin.generic.step.ArtifactCopyStep;
import com.xebialabs.deployit.plugin.generic.step.ArtifactDeleteStep;
import com.xebialabs.deployit.plugin.generic.step.SuccessStep;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.collect.Maps.newHashMap;
import static java.lang.String.format;

@SuppressWarnings("serial")
@Metadata(virtual = true, description = "An artifact deployed on a generic container")
@Placeholders
public class CopiedArtifact<D extends DeployableArtifact> extends AbstractDeployedArtifact<D> implements DerivedArtifact<D> {

    private OverthereFile placeholderProcessedFile;

    private boolean useDescriptionGeneratedByStep = false;

    @Property(defaultValue = "false", required = false, hidden = true, description = "If set to true, files are directly copied to the target directory without first being uploaded to the staging or temporary directory (like versions of XL Deploy prior to 4.0.0)" )
    private boolean copyDirectlyToTargetPath;

    @Property(defaultValue = "false", required = false, hidden = true, description = "If set to true, preserve the existing files on the remote host (do not delete the copied files during the destroy operation). ")
    private boolean preserveExistingFiles;

    @Property(defaultValue = "false", required = false, hidden = true, description = "If set to true, the sub directories on the target machine are not deleted if files other than that copied by XL Deploy are present. Please note that setting this option to true will cause the removal process to be a slower. ")
    private boolean targetPathSharedSubDirectories;

    @Property(description = "A Map containing all the placeholders mapped to their values. Special values are &lt;ignore&gt; or &lt;empty&gt;", required = false)
    private Map<String, String> placeholders = newHashMap();

    public CopiedArtifact(String id) {
        this.id = id;
    }

    public CopiedArtifact() {
    }

    @Override
    public Map<String, String> getPlaceholders() {
        return placeholders;
    }

    @Override
    public void setPlaceholders(Map<String, String> placeholders) {
        this.placeholders = placeholders;
    }

    @Override
    public OverthereFile getFile() {
        return placeholderProcessedFile;
    }

    @Override
    public void setFile(OverthereFile file) {
        this.placeholderProcessedFile = file;
    }

    @Override
    public D getSourceArtifact() {
        return getDeployable();
    }

    @Create
    public void executeCreate(DeploymentPlanningContext ctx, Delta delta) {
        Step step = createArtifactCopyStep();
        ctx.addStepWithCheckpoint(step, delta, Operation.CREATE);
    }

    protected ArtifactCopyStep createArtifactCopyStep() {
        ArtifactCopyStep step = new ArtifactCopyStep(getCreateOrder(), getFile(), getContainer(), getTargetDirectory());
        step.setSourceFileDescription(getDeployable().getName());
        if(!copyDirectlyToTargetPath) {
            step.setSourceArtifact(this);
        }
        step.setCreateTargetPath(isCreateTargetDirectory());
        step.setTargetFileName(resolveTargetFileName());
        if (!useDescriptionGeneratedByStep) {
            step.setDescription(getDescription(getCreateVerb()));
        }
        return step;
    }

    @Modify
    public void executeModify(DeploymentPlanningContext ctx, Delta delta) {
        @SuppressWarnings("unchecked")
        CopiedArtifact<D> old = (CopiedArtifact<D>) delta.getPrevious();
        old.executeDestroy(ctx, delta);
        executeCreate(ctx, delta);
    }

    @Destroy
    public void executeDestroy(DeploymentPlanningContext ctx, Delta delta) {
        final Step step;
        if (!preserveExistingFiles) {
            step = createArtifactDeleteStep();
        } else {
            step = new SuccessStep(getDestroyOrder(), format("Preserve the existing files %s on %s", getTargetDirectory(), getContainer()));
        }
        ctx.addStepWithCheckpoint(step, delta, Operation.DESTROY);
    }

    protected ArtifactDeleteStep createArtifactDeleteStep() {
        ArtifactDeleteStep step = new ArtifactDeleteStep(getDestroyOrder(), getContainer(), getDeployable(), getTargetDirectory());
        step.setTargetFile(resolveTargetFileName());
        step.setTargetDirectoryShared(isTargetDirectoryShared());
        step.setTargetPathSharedSubDirectories(isTargetPathSharedSubDirectories());
        if (!useDescriptionGeneratedByStep) {
            step.setDescription(getDescription(getDestroyVerb()));
        }
        return step;
    }

    public boolean isUseDescriptionGeneratedByStep() {
        return useDescriptionGeneratedByStep;
    }

    public void setUseDescriptionGeneratedByStep(final boolean useDescriptionGeneratedByStep) {
        this.useDescriptionGeneratedByStep = useDescriptionGeneratedByStep;
    }

    public boolean isCopyDirectlyToTargetPath() {
        return copyDirectlyToTargetPath;
    }

    public void setCopyDirectlyToTargetPath(final boolean copyDirectlyToTargetPath) {
        this.copyDirectlyToTargetPath = copyDirectlyToTargetPath;
    }

    public boolean isPreserveExistingFiles() {
        return preserveExistingFiles;
    }

    public void setPreserveExistingFiles(final boolean preserveExistingFiles) {
        this.preserveExistingFiles = preserveExistingFiles;
    }

    public boolean isTargetPathSharedSubDirectories() {
        return targetPathSharedSubDirectories;
    }

    public void setTargetPathSharedSubDirectories(final boolean targetPathSharedSubDirectories) {
        this.targetPathSharedSubDirectories = targetPathSharedSubDirectories;
    }

}
