package com.xebialabs.deployit.plugin.generic.step;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.xebialabs.deployit.plugin.api.flow.*;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.generic.freemarker.ArtifactUploader;
import com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.OperatingSystemFamily;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.RuntimeIOException;
import com.xebialabs.overthere.util.OverthereUtils;
import de.schlichtherle.truezip.file.TArchiveDetector;
import de.schlichtherle.truezip.file.TFile;
import de.schlichtherle.truezip.file.TFileInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.List;
import java.util.Map;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStderr;
import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStdout;
import static com.xebialabs.deployit.plugin.remoting.preview.PreviewOverthereConnection.getPreviewConnection;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.dumpScript;

@SuppressWarnings("serial")
public abstract class BaseExecutionStep extends BaseStep implements PreviewStep, StageableStep {

    private static final String MDC_KEY_SCRIPT_PATH = "scriptPath";

    private String scriptTemplatePath;

    private Map<String, Object> vars;

    private File file;

    private DerivedArtifact derivedArtifact;

    private StagedFile stagedDerivedArtifact;

    private List<File> fileResources = newArrayList();
    private List<String> classpathResources = newArrayList();
    private List<String> templateClasspathResources = newArrayList();

    private Preview preview;

    @SuppressWarnings("unused")
    public BaseExecutionStep(){
        vars = newHashMap();
    }

    @SuppressWarnings("unused")
    public BaseExecutionStep(String scriptPath, HostContainer container, Map<String, Object> vars, String description) {
        this(null, scriptPath, container, vars, description);
    }

    public BaseExecutionStep(Integer order, String scriptPath, HostContainer container, Map<String, Object> vars, String description) {
        super(order, description, container);
        this.scriptTemplatePath = scriptPath;
        this.vars = newHashMap(vars);
        Preconditions.checkNotNull(scriptTemplatePath);
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(BaseExecutionStep.class);
    }

    @Override
    public void requestStaging(StagingContext ctx) {
        if (derivedArtifact != null) {
            getLogger().debug("[{}] is requesting staging of [{}]", this, derivedArtifact);
            stagedDerivedArtifact = ctx.stageArtifact(derivedArtifact, container.getHost());
        } else {
            getLogger().debug("[{}] is not requesting artifact staging.", this);
        }
    }

    @Override
    public StepExitCode doExecute() throws Exception {
        MDC.put(MDC_KEY_SCRIPT_PATH, scriptTemplatePath);
        try {
            uploadArtifactIfPresent();
            uploadFileResources();
            uploadClasspathResources();
            uploadTemplateClasspathResources();
            OverthereFile executable = uploadScriptToExecute(scriptTemplatePath);
            setWorkingDirectory();
            return executeScript(executable);
        } finally {
            MDC.remove(MDC_KEY_SCRIPT_PATH);
        }
    }

    @Override
    public Preview getPreview() {
        if (preview == null) {
            ctx = new PreviewExecutionContext();
            getLocalConnection();
            remoteConn = getPreviewConnection();
            try {
                String sourcePath = resolveOsSpecificTemplate(scriptTemplatePath);
                String contents = evaluateTemplate(sourcePath, vars, true);
                preview = Preview.withSourcePathAndContents(sourcePath, contents);
            } finally {
                ctx = null;
                disconnect();
            }
        }
        return preview;
    }

    protected OverthereFile uploadArtifactIfPresent() {
        if (file != null) {
            return uploadToWorkingDirectory(file, file.getName());
        }
        return null;
    }

    protected void uploadFileResources() {
        for (File fileResource : fileResources) {
            uploadToWorkingDirectory(fileResource, fileResource.getName());
        }
    }

    protected void uploadClasspathResources() {
        for (String cpResource : classpathResources) {
            URL resource = Thread.currentThread().getContextClassLoader().getResource(cpResource);
            checkArgument(resource != null, "resource %s not found.", cpResource);
            try {
                TFile file = new TFile(resource.toURI());
                uploadFileOrDirectory(getRemoteWorkingDirectory(), file, true);
            } catch (URISyntaxException | IOException e) {
                throw new RuntimeIOException(e);
            }
        }
    }

    protected void uploadFileOrDirectory(final OverthereFile remoteDir, final TFile file, boolean logFileNameToConsole) throws IOException {
        if (file.isFile() && !file.isDirectory()) {
            //upload files and archives (zip,jar,etc)
            if (logFileNameToConsole) {
                getCtx().logOutput("Uploading file " + file.getName() + " to working directory.");
            }
            final OverthereFile targetFile = remoteDir.getFile(file.getName());
            try(InputStream is = new TFileInputStream(file)) {
                OverthereUtils.write(is, targetFile);
            }
        } else {
            //upload directory
            if (logFileNameToConsole) {
                getCtx().logOutput("Uploading directory " + file.getName() + " to working directory.");
            }
            OverthereFile targetDir = remoteDir.getFile(file.getName());
            if (!targetDir.exists()) {
                targetDir.mkdir();
            }
            for (TFile childFile : file.listFiles(TArchiveDetector.NULL)) {
                uploadFileOrDirectory(targetDir, childFile, false);
            }
        }
    }

    protected void uploadTemplateClasspathResources() {
        for (String cpTemplate : templateClasspathResources) {
            uploadTemplateResourceToWorkingDirectory(cpTemplate);
        }
    }

    private OverthereFile uploadTemplateResourceToWorkingDirectory(String template) {
        String osSpecificTemplate = resolveOsSpecificTemplate(template);
        String content = evaluateTemplate(osSpecificTemplate, vars, false);

        String fileName = resolveOsSpecificFileName(osSpecificTemplate);
        return uploadToWorkingDirectory(content, fileName);
    }

    protected OverthereFile uploadScriptToExecute(String scriptTemplatePath) {
        String osSpecificTemplate = resolveOsSpecificTemplate(scriptTemplatePath);
        String generatedScript = evaluateTemplate(osSpecificTemplate, vars, false);
        if (scriptsLogger.isTraceEnabled()) {
            String obfuscatedScript = evaluateTemplate(osSpecificTemplate, vars, true);
            dumpScript(scriptTemplatePath, obfuscatedScript, scriptsLogger);
        }

        String fileName = resolveOsSpecificFileName(osSpecificTemplate);
        return uploadToWorkingDirectory(generatedScript, fileName);
    }

    protected void setWorkingDirectory() {
        getRemoteConnection().setWorkingDirectory(getRemoteWorkingDirectory());
    }

    protected StepExitCode executeScript(OverthereFile executable) {
        executable.setExecutable(true);

        CmdLine cmdLine = CmdLine.build(executable.getPath());
        try (DefaultExecutionOutputHandler stdoutHandler = handleStdout(getCtx());
             DefaultExecutionOutputHandler stderrHandler = handleStderr(getCtx())) {
            getCtx().logOutput("Executing " + executable.getPath() + " on host " + getContainer().getHost());
            int rc = getRemoteConnection().execute(stdoutHandler, stderrHandler, cmdLine);
            if (rc != 0) {
                getCtx().logError("Execution failed with return code " + rc);
                return StepExitCode.FAIL;
            }
            return StepExitCode.SUCCESS;
        }
    }

    public DerivedArtifact getDerivedArtifact() {
        return derivedArtifact;
    }

    public String getScriptPath() {
        return scriptTemplatePath;
    }

    public List<File> getFileResources() {
        return fileResources;
    }

    public void setFileResources(List<File> fileResources) {
        this.fileResources = fileResources;
    }

    public List<String> getClasspathResources() {
        return classpathResources;
    }

    public void setClasspathResources(List<String> classpathResources) {
        this.classpathResources = classpathResources;
    }

    public List<String> getTemplateClasspathResources() {
        return templateClasspathResources;
    }

    public void setTemplateClasspathResources(List<String> templateClasspathResources) {
        this.templateClasspathResources = templateClasspathResources;
    }

    public File getArtifact() {
        return file;
    }

    /**
     * Use {link #setDerivedArtifact} instead, as that supports staging.
     */
    @Deprecated
    public void setArtifact(File artifact) {
        this.file = artifact;
    }

    public void setDerivedArtifact(DerivedArtifact da) {
        this.derivedArtifact = da;
    }

    public String getUploadedArtifactPath() {
        if (file != null) {
            return getRemoteWorkingDirectory().getPath() + getContainer().getHost().getOs().getFileSeparator() + file.getName();
        } else if (derivedArtifact != null) {
            WorkingFolderUploader uploader = (WorkingFolderUploader) getArtifactUploader();
            String path = uploader.lookup(derivedArtifact);
            return path == null ? "" : path;
        }
        return "";
    }

    protected String resolveOsSpecificFileName(String template) {
        return resolveOsSpecificFileName(template, getContainer());
    }

    public static String resolveOsSpecificFileName(String template, HostContainer hostContainer) {
        String osSpecificFile = substringAfterLast(template, '/', template);
        if (osSpecificFile.endsWith(FREEMARKER_FILE_EXT))
            osSpecificFile = osSpecificFile.substring(0, osSpecificFile.lastIndexOf(FREEMARKER_FILE_EXT));

        String fileExt = substringAfterLast(osSpecificFile, '.');
        if (fileExt == null) {
            OperatingSystemFamily os = hostContainer.getHost().getOs();
            osSpecificFile = osSpecificFile + os.getScriptExtension();
        }

        return osSpecificFile;
    }

    @Override
    protected ArtifactUploader createArtifactUploader() {
        return new WorkingFolderUploader(getCtx());
    }

    private class WorkingFolderUploader implements ArtifactUploader {
        private class ArtifactCacheKey {
            Artifact artifact;
            public ArtifactCacheKey(Artifact artifact) {
                this.artifact = artifact;
            }

            @Override
            public int hashCode() { return 1; }

            @Override
            public boolean equals(Object obj) {
                return obj instanceof ArtifactCacheKey && ((ArtifactCacheKey) obj).artifact == artifact;
            }
        }
        private Map<ArtifactCacheKey, String> uploadedFiles = newHashMap();
        private ExecutionContext ctx;

        public WorkingFolderUploader(ExecutionContext ctx) {
            this.ctx = ctx;
        }

        public String lookup(Artifact artifact) {
            return uploadedFiles.get(new ArtifactCacheKey(artifact));
        }

        @Override
        public String upload(Artifact artifact) {
            ArtifactCacheKey artifactKey = new ArtifactCacheKey(artifact);
            if (uploadedFiles.containsKey(artifactKey)) {
                return uploadedFiles.get(artifactKey);
            }

            String path;
            getLogger().trace("Requesting upload of artifact [{}]", artifact);
            getLogger().trace("Derived artifact = [{}]", derivedArtifact);
            if (derivedArtifact != null && stagedDerivedArtifact != null && artifact == derivedArtifact) {
                getLogger().debug("Retrieving result of staging for artifact [{}]", derivedArtifact);
                OverthereFile uploadedDerivedArtifact = stagedDerivedArtifact.get(getRemoteConnection(), ctx);
                OverthereFile dir = getRemoteWorkingDirectory();
                ctx.logOutput("Copying " + getFileType(uploadedDerivedArtifact) + " " + uploadedDerivedArtifact.getName() + " to working directory.");
                uploadedDerivedArtifact.copyTo(dir.getFile(uploadedDerivedArtifact.getName()));
                path = dir.getFile(uploadedDerivedArtifact.getName()).getPath();
            } else {
                OverthereFile dir = OverthereUtils.getUniqueFolder(getRemoteWorkingDirectory(), artifact.getName());
                OverthereFile uploadedFile = dir.getFile(artifact.getName());
                if (!uploadedFile.exists()) {
                    OverthereFile file = artifact.getFile();
                    String fileType = getFileType(file);
                    ctx.logOutput("Uploading " + fileType + " " + artifact.getName() + " to working directory.");
                    file.copyTo(uploadedFile);
                } else {
                    getLogger().warn("Not uploading file " + artifact.getName() + " to working directory again because it has already been uploaded.");
                }
                path = uploadedFile.getPath();
            }

            uploadedFiles.put(artifactKey, path);
            return path;
        }

        private String getFileType(OverthereFile file) {
            return file.isDirectory() ? "directory" : "file";
        }
    }


    @VisibleForTesting
    public String getScriptTemplatePath() {
        return scriptTemplatePath;
    }

    @VisibleForTesting
    public Map<String, Object> getVars() {
        return vars;
    }

    private static final Logger scriptsLogger = LoggerFactory.getLogger("com.xebialabs.deployit.plugin.generic.scripts");

}
