package com.xebialabs.deployit.plugin.generic.freemarker;

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;

import static com.google.common.collect.Lists.newArrayList;

import static com.google.common.collect.Sets.newHashSet;

import freemarker.ext.beans.BeanModel;
import freemarker.template.TemplateModel;
import freemarker.template.TemplateModelException;

import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder.resolveExpression;

public class CiTemplateModel extends BeanModel {

    private static final String FILE_ATTRIBUTE = "file";
    public static final String MASKED_PASSWORD = "********";

    private ConfigurationItem ci;
    private CiAwareObjectWrapper wrapper;
    private boolean maskPasswords;

    public CiTemplateModel(ConfigurationItem ci, CiAwareObjectWrapper wrapper, boolean maskPasswords) {
        super(ci, wrapper);
        this.ci = ci;
        this.wrapper = wrapper;
        this.maskPasswords = maskPasswords;
    }

    @Override
    public TemplateModel get(String key) throws TemplateModelException {
        if (key.equals(FILE_ATTRIBUTE)) {
            if (ci instanceof SourceArtifact) {
                return handleSourceArtifact();
            } else if (ci instanceof DerivedArtifact) {
                return handleDerivedArtifact();
            }
        }

        PropertyDescriptor pd = ci.getType().getDescriptor().getPropertyDescriptor(key);
        if (pd == null) {
            return super.get(key);
        } else {
            return wrapper.wrap(extractValueFromCi(pd));
        }
    }

    @SuppressWarnings("unchecked")
    private Object extractValueFromCi(PropertyDescriptor pd) {
        Map<String, Object> ctx = newHashMap();
        if (ci instanceof com.xebialabs.deployit.plugin.api.udm.Deployed) {
            ctx.put("deployed", ci);
        } else if (ci instanceof com.xebialabs.deployit.plugin.api.udm.Container) {
            ctx.put("container", ci);
        }

        if (pd.isPassword() && maskPasswords) {
            switch (pd.getKind()) {
                case SET_OF_STRING:
                    return newHashSet(MASKED_PASSWORD);
                case LIST_OF_STRING:
                    return newArrayList(MASKED_PASSWORD);
                case MAP_STRING_STRING:
                    Map<String, String> maskedMap = newHashMap();
                    maskedMap.put(MASKED_PASSWORD, MASKED_PASSWORD);
                    return maskedMap;
                default:
                    return MASKED_PASSWORD;
            }
        }

        switch (pd.getKind()) {
            case STRING:
                return resolveExpression((String) pd.get(ci), ctx, maskPasswords);
            case SET_OF_STRING:
                return resolveExpression((Set<String>) pd.get(ci), ctx, maskPasswords);
            case LIST_OF_STRING:
                return resolveExpression((List<String>) pd.get(ci), ctx, maskPasswords);
            case MAP_STRING_STRING:
                return resolveExpression((Map<String, String>) pd.get(ci), ctx, maskPasswords);
            default:
                return pd.get(ci);
        }
    }

    private TemplateModel handleSourceArtifact() throws TemplateModelException {
        return uploadArtifactIfUploaderPresent(((SourceArtifact) ci));
    }

    private TemplateModel handleDerivedArtifact() throws TemplateModelException {
        return uploadArtifactIfUploaderPresent(((DerivedArtifact<?>) ci));
    }

    private TemplateModel uploadArtifactIfUploaderPresent(Artifact artifact) throws TemplateModelException {
        String path = artifact.getFile().getPath();
        if (wrapper.getUploader() != null) {
            path = wrapper.getUploader().upload(artifact);
        }
        return wrapper.wrap(path);
    }

    @Override
    public boolean isEmpty() {
        return false;
    }
}
