package com.xebialabs.deployit.plugin.generic.step;

import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.io.ByteStreams;
import com.google.common.io.Resources;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.generic.freemarker.ArtifactUploader;
import com.xebialabs.deployit.plugin.generic.freemarker.CiAwareObjectWrapper;
import com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.deployit.plugin.steps.CalculatedStep;
import com.xebialabs.overthere.OperatingSystemFamily;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.RuntimeIOException;
import com.xebialabs.overthere.local.LocalConnection;
import com.xebialabs.overthere.local.LocalFile;
import com.xebialabs.overthere.util.OverthereUtils;
import com.xebialabs.xlplatform.satellite.Satellite;
import com.xebialabs.xlplatform.satellite.SatelliteAware;
import freemarker.ext.beans.BeansWrapper;
import freemarker.ext.dom.NodeModel;
import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import javax.xml.parsers.ParserConfigurationException;
import java.io.*;
import java.net.URL;
import java.util.Map;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Strings.emptyToNull;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.booter.local.utils.Closeables.closeQuietly;

@SuppressWarnings("serial")
public abstract class BaseStep extends CalculatedStep implements SatelliteAware {

    public static final String FREEMARKER_FILE_EXT = ".ftl";

    protected HostContainer container;

    private String remoteWorkingDirPath;
    private boolean retainRemoteWorkingDirOnCompletion;

    private transient OverthereConnection localConn;
    protected transient OverthereConnection remoteConn;
    protected transient ExecutionContext ctx;
    private transient OverthereFile remoteWorkingDir;
    private transient ArtifactUploader artifactUploader;

    protected BaseStep() {

    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(BaseStep.class);
    }

    @SuppressWarnings("unused")
    protected BaseStep(HostContainer container) {
        this(null, container);
    }

    protected BaseStep(String description, HostContainer container) {
        this(null, description, container);
    }

    protected BaseStep(Integer order, String description, HostContainer container) {
        super(order, description);
        this.container = container;
    }

    public StepExitCode handleExecute(ExecutionContext ctx) throws Exception {
        try {
            this.ctx = ctx;
            return doExecute();
        } finally {
            disconnect();
        }
    }

    protected abstract StepExitCode doExecute() throws Exception;

    protected ExecutionContext getCtx() {
        return ctx;
    }

    public HostContainer getContainer() {
        return container;
    }

    public OverthereFile getRemoteWorkingDirectory() {
        if (remoteWorkingDir == null) {
            OverthereFile tempDir;
            if (Strings.isNullOrEmpty(getRemoteWorkingDirPath())) {
                tempDir = getRemoteConnection().getTempFile("generic_plugin", ".tmp");
            } else {
                tempDir = getRemoteConnection().getFile(getRemoteWorkingDirPath());
            }
            tempDir.mkdir();
            remoteWorkingDir = tempDir;
        }
        return remoteWorkingDir;
    }


    public OverthereConnection getLocalConnection() {
        if (localConn == null) {
            localConn = LocalConnection.getLocalConnection();
        }
        return localConn;
    }

    public OverthereConnection getRemoteConnection() {
        if (remoteConn == null) {
            remoteConn = getContainer().getHost().getConnection();
        }
        return remoteConn;
    }

    protected void disconnect() {
        closeQuietly(localConn);

        if (!Strings.isNullOrEmpty(getRemoteWorkingDirPath()) && !isRetainRemoteWorkingDirOnCompletion()) {
            getRemoteWorkingDirectory().deleteRecursively();
        }

        closeQuietly(remoteConn);

        remoteWorkingDir = null;
        localConn = null;
        remoteConn = null;
        artifactUploader = null;
    }

    public ArtifactUploader getArtifactUploader() {
        if (artifactUploader == null) {
            artifactUploader = createArtifactUploader();
        }
        return artifactUploader;
    }

    public String getRemoteWorkingDirPath() {
        return remoteWorkingDirPath;
    }

    public void setRemoteWorkingDirPath(String remoteWorkingDirPath) {
        this.remoteWorkingDirPath = remoteWorkingDirPath;
    }

    public boolean isRetainRemoteWorkingDirOnCompletion() {
        return retainRemoteWorkingDirOnCompletion;
    }

    public void setRetainRemoteWorkingDirOnCompletion(boolean deleteWorkingDirOnCompletion) {
        this.retainRemoteWorkingDirOnCompletion = deleteWorkingDirOnCompletion;
    }

    protected OverthereFile uploadToWorkingDirectory(String content, String fileName) {
        getCtx().logOutput("Uploading file " + fileName + " to working directory.");
        OverthereFile target = getRemoteWorkingDirectory().getFile(fileName);
        OverthereUtils.write(content.getBytes(), target);
        return target;
    }

    protected OverthereFile uploadToWorkingDirectory(File content, String fileName) {
        String fileType = content.isDirectory() ? "directory" : "file";
        getCtx().logOutput("Uploading " + fileType + " " + fileName + " to working directory.");
        OverthereFile target = getRemoteWorkingDirectory().getFile(fileName);
        LocalFile.valueOf(content).copyTo(target);
        return target;
    }

    protected OverthereFile uploadToWorkingDirectory(URL content, String fileName) {
        getCtx().logOutput("Uploading file " + fileName + " to working directory.");
        OverthereFile target = getRemoteWorkingDirectory().getFile(fileName);
        try (OutputStream out = target.getOutputStream()) {
            Resources.copy(content, out);
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        }
        return target;
    }

    public boolean hostFileExists(String remoteFile) {
        checkNotNull(emptyToNull(remoteFile));
        OverthereFile file = getRemoteConnection().getFile(remoteFile);
        return file.exists();
    }

    public String getHostLineSeparator() {
        return getContainer().getHost().getOs().getLineSeparator();
    }

    public String getHostFileSeparator() {
        return getContainer().getHost().getOs().getFileSeparator();
    }

    public String readHostFile(final String remoteFile) {
        checkNotNull(emptyToNull(remoteFile));
        OverthereFile file = getRemoteConnection().getFile(remoteFile);
        checkArgument(file.exists(), "File %s does not exist on host %s", remoteFile, getContainer().getHost());
        try (InputStream in = file.getInputStream()) {
            byte[] bytes = ByteStreams.toByteArray(in);
            return new String(bytes);
        } catch (IOException e) {
            throw new RuntimeIOException("Failed to read file " + remoteFile, e);
        }
    }

    public String[] readHostFileLines(final String remoteFile) {
        String data = readHostFile(remoteFile);
        Iterable<String> iterable = Splitter.on(getHostLineSeparator()).split(data);
        return Iterables.toArray(iterable, String.class);
    }

    public void createOrReplaceHostFile(String content, String remoteFile) {
        checkNotNull(emptyToNull(remoteFile));
        checkNotNull(content);
        OverthereFile file = getRemoteConnection().getFile(remoteFile);
        if (file.exists()) {
            getLogger().debug("File " + remoteFile + " already exists. Will delete before attempting to write.");
            file.delete();
        }

        if (!file.getParentFile().exists()) {
            getLogger().debug("Parent directory does not exist. Will create it.");
            file.getParentFile().mkdirs();
        }


        try (final ByteArrayInputStream from = new ByteArrayInputStream(content.getBytes());
             final OutputStream to = file.getOutputStream()) {
            ByteStreams.copy(from, to);
        } catch (IOException e) {
            throw new RuntimeIOException("Failed to write to " + remoteFile, e);
        }
    }

    public NodeModel readHostXmlFileAsModel(final String remoteXmlFile) {
        checkNotNull(emptyToNull(remoteXmlFile));
        OverthereFile file = getRemoteConnection().getFile(remoteXmlFile);
        checkArgument(file.exists(), "File %s does not exist on host %s", remoteXmlFile, getContainer().getHost());

        try (InputStream remoteXmlStream = file.getInputStream()) {
            return NodeModel.parse(new InputSource(remoteXmlStream), false, true);
        } catch (ParserConfigurationException | SAXException exc) {
            throw new RuntimeException("Cannot read xml file " + remoteXmlFile, exc);
        } catch (IOException exc) {
            throw new RuntimeIOException("Cannot read xml file " + remoteXmlFile, exc);
        }
    }

    public String evaluateTemplate(String templatePath, Map<String, Object> vars, boolean maskPasswords) {
        return evaluateTemplate(templatePath, vars, maskPasswords, getArtifactUploader());
    }

    public String evaluateTemplate(String templatePath, Map<String, Object> vars, boolean maskPasswords, ArtifactUploader artifactUploader) {
        Configuration cfg = ConfigurationHolder.getConfiguration();
        try {
            Template template = cfg.getTemplate(templatePath);
            StringWriter sw = new StringWriter();

            Map<String, Object> varsWithStatics = enrichedTemplateVariables(vars);
            template.createProcessingEnvironment(varsWithStatics, sw, new CiAwareObjectWrapper(artifactUploader, maskPasswords)).process();
            return sw.toString();
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        } catch (TemplateException e) {
            throw new RuntimeException(e);
        }
    }

    protected ArtifactUploader createArtifactUploader() {
        return new ArtifactUploader() {
            @Override
            public String upload(Artifact file) {
                throw new UnsupportedOperationException("Artifact upload not supported in this step");
            }
        };
    }

    protected void evaluateTemplate(OverthereFile renderTo, String templatePath, Map<String, Object> vars) {
        Configuration cfg = ConfigurationHolder.getConfiguration();
        try (OutputStream out = renderTo.getOutputStream()) {
            Map<String, Object> variables = enrichedTemplateVariables(vars);
            Template template = cfg.getTemplate(templatePath);
            template.process(variables, new OutputStreamWriter(out));
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        } catch (TemplateException e) {
            throw new RuntimeException(e);
        }
    }

    private Map<String, Object> enrichedTemplateVariables(final Map<String, Object> vars) {
        Map<String, Object> variables = newHashMap(vars);
        variables.put("step", this);
        variables.put("context", this.getCtx());
        variables.put("statics", BeansWrapper.getDefaultInstance().getStaticModels());
        return variables;
    }

    public String resolveOsSpecificTemplate(String template) {
        if (classpathResourceExists(template + FREEMARKER_FILE_EXT)) {
            return template + FREEMARKER_FILE_EXT;
        }

        String osSpecificTemplate = template;

        String scriptExt = substringAfterLast(osSpecificTemplate, '.');
        if (scriptExt == null) {
            OperatingSystemFamily os = getContainer().getHost().getOs();
            osSpecificTemplate = osSpecificTemplate + os.getScriptExtension();
        }

        if (!classpathResourceExists(osSpecificTemplate)) {
            String osSpecificScriptTemplate = osSpecificTemplate + FREEMARKER_FILE_EXT;
            if (!classpathResourceExists(osSpecificScriptTemplate))
                throw new IllegalArgumentException("Resource " + osSpecificTemplate + " not found in classpath");
            else
                osSpecificTemplate = osSpecificScriptTemplate;
        }

        return osSpecificTemplate;
    }

    public boolean classpathResourceExists(String resource) {
        return Thread.currentThread().getContextClassLoader().getResource(resource) != null;
    }

    @Override
    public Satellite getSatellite() {
        return container.getHost().getSatellite();
    }

    public static String substringAfterLast(String str, char sub, String defaultValue) {
        String s = substringAfterLast(str, sub);
        if (s == null) {
            return defaultValue;
        }
        return s;
    }

    public static String substringAfterLast(String str, char sub) {
        int pos = str.lastIndexOf(sub);
        if (pos == -1) {
            return null;
        } else {
            return str.substring(pos + 1);
        }
    }


}
