package com.xebialabs.deployit.plugin.generic.deployed;

import java.util.List;
import java.util.Map;

import com.google.common.base.Strings;

import com.xebialabs.deployit.plugin.api.deployment.planning.Create;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.planning.Destroy;
import com.xebialabs.deployit.plugin.api.deployment.planning.Modify;
import com.xebialabs.deployit.plugin.api.deployment.planning.Noop;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.generic.step.InstructionStep;
import com.xebialabs.deployit.plugin.mail.SmtpServer;

import static com.google.common.collect.Maps.newHashMap;

import static com.google.common.collect.Lists.newArrayList;

@SuppressWarnings("serial")
@Metadata(virtual = true, description = "A manual process that needs to be performed on a generic container")
public class ManualProcess<D extends Deployable> extends AbstractDeployed<D> {

    @Property(required = false, description = "Mail addresses of recepients.", category = "Mail")
    private List<String> toAddresses = newArrayList();

    @Property(required = false, description = "Mail subject", category = "Mail")
    private String subject;

    @Property(required = false, description = "From mail address. Defaults to SMTPServer fromAddress.", category = "Mail")
    private String fromAddress;

    @Property(required = true, hidden = true, description = "Classpath to the script that generates the instructions for the create operation.")
    private String createScript;

    @Property(required = false, hidden = true, description = "Classpath to the script that generates the instructions for the modify operation.")
    private String modifyScript;

    @Property(required = false, hidden = true, description = "Classpath to the script that generates the instructions for the destroy operation.")
    private String destroyScript;

    @Property(required = false, hidden = true, description = "Classpath to the script that generates the instructions for the noop operation.")
    private String noopScript;

    protected boolean addStep(DeploymentPlanningContext ctx, int order, String script, String verb) {
        return addStep(ctx, order, script, verb, null);
    }

    protected boolean addStep(DeploymentPlanningContext ctx, int order, String script, String verb, CheckpointInfo checkpoint) {
        return addStep(ctx, order, script, verb, checkpoint, null);
    }

    protected boolean addStep(DeploymentPlanningContext ctx, int order, String script, String verb, CheckpointInfo checkpoint, Deployed<?, ?> previousDeployed) {
        if (!Strings.nullToEmpty(script).trim().isEmpty()) {
            Map<String, Object> freeMarkerContext = getDeployedAsFreeMarkerContext();
            if(previousDeployed != null) {
                freeMarkerContext = newHashMap(freeMarkerContext);
                freeMarkerContext.put("previousDeployed", previousDeployed);
            }

            InstructionStep step = new InstructionStep(order, getDescription(verb),getContainer(), freeMarkerContext, script);
            step.setFromAddress(getFromAddress());
            step.setMailServer(getMailServer(ctx));
            step.setSubject(getSubject());
            step.setToAddresses(getToAddresses());

            if (checkpoint == null) {
                ctx.addStep(step);
            } else {
                ctx.addStepWithCheckpoint(step, checkpoint.delta, checkpoint.operation);
            }
            return true;
        } else {
            return false;
        }
    }

    protected SmtpServer getMailServer(DeploymentPlanningContext ctx) {
        return SmtpServer.getMailServer(ctx);
    }

    @SuppressWarnings("unchecked")
    @Create
    public void executeCreate(DeploymentPlanningContext ctx, Delta delta) {
        ManualProcess<D> script = (ManualProcess<D>) delta.getDeployed();
        script.create(ctx, delta);
    }

    protected void create(DeploymentPlanningContext ctx, Delta delta) {
        addStep(ctx, getCreateOrder(), getCreateScript(), getCreateVerb(), checkpoint(delta, Operation.CREATE));
    }

    @Modify
    public void executeModify(DeploymentPlanningContext ctx, Delta delta) {
        boolean modifyStepAdded = addStep(ctx, getModifyOrder(), getModifyScript(), getModifyVerb(), checkpoint(delta, Operation.MODIFY), delta.getPrevious());
        if (!modifyStepAdded) {
            executeDestroy(ctx, delta);
            executeCreate(ctx, delta);
        }
    }

    @SuppressWarnings("unchecked")
    @Destroy
    public void executeDestroy(DeploymentPlanningContext ctx, Delta delta) {
        ManualProcess<D> previous = (ManualProcess<D>) delta.getPrevious();
        previous.destroy(ctx, delta);
    }

    protected void destroy(DeploymentPlanningContext ctx, Delta delta) {
        addStep(ctx, getDestroyOrder(), getDestroyScript(), getDestroyVerb(), checkpoint(delta, Operation.DESTROY));
    }

    @Noop
    public void executeNoop(DeploymentPlanningContext ctx, Delta delta) {
        addStep(ctx, getNoopOrder(), getNoopScript(), getNoopVerb());
    }

    public String getCreateScript() {
        return resolveExpression(createScript);
    }

    public void setCreateScript(String createScript) {
        this.createScript = createScript;
    }

    public String getModifyScript() {
        return resolveExpression(modifyScript);
    }

    public void setModifyScript(String modifyScript) {
        this.modifyScript = modifyScript;
    }

    public String getDestroyScript() {
        return resolveExpression(destroyScript);
    }

    public void setDestroyScript(String destroyScript) {
        this.destroyScript = destroyScript;
    }

    public String getNoopScript() {
        return resolveExpression(noopScript);
    }

    public void setNoopScript(String noopScript) {
        this.noopScript = noopScript;
    }

    public List<String> getToAddresses() {
        return resolveExpression(toAddresses);
    }

    public void setToAddresses(List<String> toAddresses) {
        this.toAddresses = toAddresses;
    }

    public String getSubject() {
        return resolveExpression(subject);
    }

    public void setSubject(String subject) {
        this.subject = subject;
    }

    public String getFromAddress() {
        return resolveExpression(fromAddress);
    }

    public void setFromAddress(String fromAddress) {
        this.fromAddress = fromAddress;
    }

    protected static CheckpointInfo checkpoint(Delta delta) {
        return new CheckpointInfo(delta);
    }

    protected static CheckpointInfo checkpoint(Delta delta, Operation operation) {
        return new CheckpointInfo(delta, operation);
    }

    protected static class CheckpointInfo {
        public final Delta delta;
        public final Operation operation;

        CheckpointInfo(Delta delta) {
            this.delta = delta;
            this.operation = delta.getOperation();
        }

        CheckpointInfo(Delta delta, Operation operation) {
            this.delta = delta;
            this.operation = operation;
        }
    }
}