package com.xebialabs.deployit.plugin.mail;

import java.util.List;
import java.util.Map;
import java.util.Properties;
import javax.mail.*;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import com.google.common.base.Joiner;
import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;

import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ControlTask;
import com.xebialabs.deployit.plugin.api.udm.Environment;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Strings.isNullOrEmpty;
import static com.google.common.base.Strings.nullToEmpty;
import static com.google.common.collect.Iterables.tryFind;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.CONFIGURATION;
import static java.lang.String.format;

@SuppressWarnings("serial")
@Metadata(root = CONFIGURATION, description = "SMTP Mail Server Configuration")
public class SmtpServer extends BaseConfigurationItem {

    public static final String DEFAULT_SMTP_SERVER_NAME = "defaultSmtpServer";

    @Property(description = "SMTP host")
    private String host;

    @Property(description = "SMTP port", defaultValue = "25")
    private int port;

    @Property(required = false, description = "Username to authenticate with host")
    private String username;

    @Property(required = false, password = true, description = "Password to authenticate with host")
    private String password;

    @Property(description = "Default from address to use for messages sent with this server.")
    private String fromAddress;

    @Property(required = false, description = "The address to which a test mail is sent when using the 'Send Test Mail' control task.")
    private String testAddress;


    @Property(required = false, description = "Refer to http://javamail.kenai.com/nonav/javadocs/com/sun/mail/smtp/package-summary.html for all properties that can be used.")
    private Map<String, String> smtpProperties = newHashMap();

    private transient Session session;

    public void sendMessage(String subject, String msg, List<String> toAddresses, String fromAddress) throws MessagingException {
        sendMessage(subject, msg, toAddresses, fromAddress,false);
    }

    public void sendMessage(String subject, String msg, List<String> toAddresses, String fromAddress, boolean isHtmlMsg) throws MessagingException {
        session = getSession();

        MimeMessage message = new MimeMessage(session);
        String addresses = Joiner.on(',').join(toAddresses);
        InternetAddress[] internetAddresses = InternetAddress.parse(addresses);
        message.setRecipients(Message.RecipientType.TO, internetAddresses);
        message.setSubject(subject);
        if (!Strings.isNullOrEmpty(fromAddress)) {
            message.setFrom(InternetAddress.parse(fromAddress)[0]);
        }
        if (isHtmlMsg) {
            message.setText(msg, null, "html");
        } else {
            message.setText(msg);
        }

        Transport.send(message);
    }

    @ControlTask
    public List<Step> sendTestMail() {
        checkArgument(!isNullOrEmpty(testAddress), "Please fill in the testAddress property before trying to send a test mail.");
        Step step = new Step() {
            @Override
            public int getOrder() {
                return 1;
            }

            @Override
            public String getDescription() {
                return "Sending test mail to " + testAddress;
            }

            @Override
            public StepExitCode execute(ExecutionContext ctx) throws Exception {
                ctx.logOutput("Sending test mail...");
                sendMessage("Test Mail", "Test message from XL Deploy.", newArrayList(testAddress),null);
                ctx.logOutput("Sent successfully.");
                return StepExitCode.SUCCESS;
            }
        };
        return newArrayList(step);
    }

    private Session getSession() {
        if (session == null) {
            session = Session.getInstance(getSessionProperties(), new Authenticator() {
                protected PasswordAuthentication getPasswordAuthentication() {
                    return new PasswordAuthentication(nullToEmpty(username), nullToEmpty(password));
                }
            });
        }
        return session;
    }

    private Properties getSessionProperties() {
        Properties props = new Properties();

        String authEnabled = "true";
        if (Strings.isNullOrEmpty(username) || Strings.isNullOrEmpty(password)) {
            authEnabled = "false";
        }

        props.put("mail.smtp.auth", authEnabled);
        props.put("mail.smtp.host", host);
        props.put("mail.smtp.port", port);
        props.put("mail.smtp.from", fromAddress);

        props.putAll(smtpProperties);
        return props;
    }

    public static SmtpServer getMailServer(DeploymentPlanningContext ctx) {
        Environment environment = ctx.getDeployedApplication().getEnvironment();
        SmtpServer smtpServer = environment.getProperty("smtpServer");

        if (smtpServer == null) {
            smtpServer = findDefaultSmtpServer(ctx);
        }

        if (smtpServer == null) {
            throw new IllegalArgumentException(format("No default SmtpServer [%s] found under Configuration root, and Environment [%s] does not have an SmtpServer configured.", SmtpServer.DEFAULT_SMTP_SERVER_NAME, ctx.getDeployedApplication().getEnvironment().getId()));
        }

        return smtpServer;
    }

    private static SmtpServer findDefaultSmtpServer(DeploymentPlanningContext ctx) {
        List<SmtpServer> search = ctx.getRepository().search(Type.valueOf(SmtpServer.class));
        Optional<SmtpServer> configurationItemOptional = tryFind(search, new Predicate<SmtpServer>() {
            @Override
            public boolean apply(SmtpServer input) {
                return input.getName().equals(SmtpServer.DEFAULT_SMTP_SERVER_NAME);
            }
        });

        return configurationItemOptional.orNull();
    }

    public String getHost() {
        return host;
    }

    public void setHost(String host) {
        this.host = host;
    }

    public int getPort() {
        return port;
    }

    public void setPort(int port) {
        this.port = port;
    }

    public String getUsername() {
        return username;
    }

    public void setUsername(String username) {
        this.username = username;
    }

    public String getPassword() {
        return password;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    public String getFromAddress() {
        return fromAddress;
    }

    public void setFromAddress(String fromAddress) {
        this.fromAddress = fromAddress;
    }

    public Map<String, String> getSmtpProperties() {
        return smtpProperties;
    }

    public void setSmtpProperties(Map<String, String> smtpProperties) {
        this.smtpProperties = smtpProperties;
    }
}
