package com.xebialabs.deployit.plugin.generic.freemarker;

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.generic.ci.Container;
import com.xebialabs.deployit.plugin.generic.ci.NestedContainer;
import com.xebialabs.deployit.plugin.generic.deployed.AbstractDeployed;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.collect.Lists.newArrayList;

import static com.google.common.collect.Sets.newHashSet;

import freemarker.ext.beans.BeanModel;
import freemarker.template.TemplateModel;
import freemarker.template.TemplateModelException;

import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder.resolveExpression;

public class CiTemplateModel extends BeanModel {

    private static final String FILE_ATTRIBUTE = "file";
    private static final String MASKED_PASSWORD = "********";

    private ConfigurationItem ci;
    private CiAwareObjectWrapper wrapper;
    private boolean maskPasswords;

    /**
     * @deprecated Use {@link #CiTemplateModel(ConfigurationItem, CiAwareObjectWrapper, boolean)} instead.
     */
    public CiTemplateModel(ConfigurationItem ci, CiAwareObjectWrapper wrapper) {
        this(ci, wrapper, false);
    }

    public CiTemplateModel(ConfigurationItem ci, CiAwareObjectWrapper wrapper, boolean maskPasswords) {
        super(ci, wrapper);
        this.ci = ci;
        this.wrapper = wrapper;
        this.maskPasswords = maskPasswords;
    }

    @Override
    public TemplateModel get(String key) throws TemplateModelException {
        if (key.equals(FILE_ATTRIBUTE)) {
            if (ci instanceof DeployableArtifact) {
                return handleDeployableArtifactFile();
            } else if (ci instanceof DerivedArtifact) {
                return handleDerivedArtifactFile();
            }
        }

        PropertyDescriptor pd = ci.getType().getDescriptor().getPropertyDescriptor(key);
        if (pd == null) {
            return super.get(key);
        } else {
            return wrapper.wrap(extractValueFromCi(pd));
        }
    }

    @SuppressWarnings("unchecked")
    private Object extractValueFromCi(PropertyDescriptor pd) {
        Map<String, Object> ctx = newHashMap();
        if (ci instanceof AbstractDeployed) {
            ctx.put("deployed", ci);
        } else if (ci instanceof Container || ci instanceof NestedContainer) {
            ctx.put("container", ci);
        }

        if (pd.isPassword() && maskPasswords) {
            switch (pd.getKind()) {
            case SET_OF_STRING:
                return newHashSet(MASKED_PASSWORD);
            case LIST_OF_STRING:
                return newArrayList(MASKED_PASSWORD);
            case MAP_STRING_STRING:
                Map<String, String> maskedMap = newHashMap();
                maskedMap.put(MASKED_PASSWORD, MASKED_PASSWORD);
                return maskedMap;
            default:
                return MASKED_PASSWORD;
            }
        }

        switch (pd.getKind()) {
            case STRING:
                return resolveExpression((String) pd.get(ci), ctx);
            case SET_OF_STRING:
                return resolveExpression((Set<String>) pd.get(ci), ctx);
            case LIST_OF_STRING:
                return resolveExpression((List<String>) pd.get(ci), ctx);
            case MAP_STRING_STRING:
                return resolveExpression((Map<String, String>) pd.get(ci), ctx);
            default:
                return pd.get(ci);
        }
    }

    private TemplateModel handleDeployableArtifactFile() throws TemplateModelException {
       return uploadFile(((DeployableArtifact) ci).getFile());
    }

    private TemplateModel handleDerivedArtifactFile() throws TemplateModelException {
        return uploadFile(((DerivedArtifact<?>) ci).getFile());
    }

    private TemplateModel uploadFile(OverthereFile file) throws TemplateModelException {
        Object o = file;
        if (wrapper.getUploader() != null) {
            o = wrapper.getUploader().upload(file);
        }
        return wrapper.wrap(o);
    }

    @Override
    public boolean isEmpty() {
        return false;
    }
}
