package com.xebialabs.deployit.plugin.generic.ci;

import com.xebialabs.deployit.plugin.api.execution.Step;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseContainer;
import com.xebialabs.deployit.plugin.generic.step.ScriptExecutionStep;
import com.xebialabs.deployit.plugin.overthere.Host;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder.resolveExpression;

@SuppressWarnings("serial")
@Metadata(virtual = true, description = "A container to which generic CIs can be deployed. Start, stop and restart behavior of this container can be controlled using the corresponding script properties.")
public class Container extends BaseContainer implements GenericContainer {

    protected Map<String, Object> freeMarkerContext = Collections.singletonMap("container", (Object) this);

    @Property(asContainment = true, description = "Host upon which the container resides")
    private Host host;

    @Property(required = false, hidden = true, description = "Classpath to the script used to start the generic container.")
    private String startScript;

    @Property(required = false, hidden = true, description = "Classpath to the script used to stop the generic container.")
    private String stopScript;

    @Property(required = false, hidden = true, description = "Classpath to the script used to restart the generic container.")
    private String restartScript;

    @Property(defaultValue = "90", hidden = true, description = "The order of the start container step in the step list.")
    private int startOrder;

    @Property(defaultValue = "10", hidden = true, description = "The order of the stop container step in the step list.")
    private int stopOrder;

    @Property(defaultValue = "90", hidden = true, description = "The order of the restart container step in the step list.")
    private int restartOrder;

    @Property(required = false, defaultValue = "0", hidden = true, description = "The time to wait in seconds for a container restart action.")
    private int restartWaitTime;

    @Property(defaultValue = "0", hidden = true, description = "The time to wait in seconds for a container start action.")
    private int startWaitTime;

    @Property(defaultValue = "0", hidden = true, description = "The time to wait in seconds for a container stop action.")
    private int stopWaitTime;

    @Property(required = false, description = "Environment variables for container", label="Environment Variables")
    private Map<String, String> envVars = newHashMap();

    @SuppressWarnings("rawtypes")
	public List<Step> controlTaskDispatch(String name) {
    	String scriptPropertyName = name + "Script";
    	PropertyDescriptor propertyDescriptor = DescriptorRegistry.getDescriptor(getType()).getPropertyDescriptor(scriptPropertyName);
    	checkArgument(propertyDescriptor!= null, "Control task script property %s not defined for CI type %s", scriptPropertyName, getType());
    	String scriptName = resolveExpression((String) propertyDescriptor.get(this), freeMarkerContext);
    	
    	return Collections.<Step>singletonList(new ScriptExecutionStep(1, scriptName, this,
    			freeMarkerContext, "Executing " + name));
    }

    public Map<String, String> getEnvVars() {
         return resolveExpression(envVars, freeMarkerContext);
    }

    public void setEnvVars(Map<String, String> envVars) {
        this.envVars = envVars;
    }

    @Override
    public Host getHost() {
        return host;
    }

    @Override
    public <T extends GenericContainer> T getParentContainer() {
        // The top Level Container does not have a parent
        return null;
    }


    public void setHost(Host host) {
        this.host = host;
    }

    public String getStartScript() {
        return resolveExpression(startScript, freeMarkerContext);
    }

    public void setStartScript(String startScript) {
        this.startScript = startScript;
    }

    public String getStopScript() {
        return resolveExpression(stopScript, freeMarkerContext);
    }

    public void setStopScript(String stopScript) {
        this.stopScript = stopScript;
    }

    public String getRestartScript() {
        return resolveExpression(restartScript, freeMarkerContext);
    }

    public void setRestartScript(String restartScript) {
        this.restartScript = restartScript;
    }

    public int getStartOrder() {
        return startOrder;
    }

    public void setStartOrder(int startOrder) {
        this.startOrder = startOrder;
    }

    public int getStopOrder() {
        return stopOrder;
    }

    public void setStopOrder(int stopOrder) {
        this.stopOrder = stopOrder;
    }

    public int getRestartOrder() {
        return restartOrder;
    }

    public void setRestartOrder(int restartOrder) {
        this.restartOrder = restartOrder;
    }

    public int getRestartWaitTime() {
        return restartWaitTime;
    }

    public void setRestartWaitTime(int restartWaitTime) {
        this.restartWaitTime = restartWaitTime;
    }

    public int getStartWaitTime() {
        return startWaitTime;
    }

    public void setStartWaitTime(int startWaitTime) {
        this.startWaitTime = startWaitTime;
    }

    public int getStopWaitTime() {
        return stopWaitTime;
    }

    public void setStopWaitTime(int stopWaitTime) {
        this.stopWaitTime = stopWaitTime;
    }

}
