package com.xebialabs.deployit.plugin.generic.step;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.Map;

import org.slf4j.MDC;

import com.google.common.base.Preconditions;
import com.google.common.io.ByteStreams;
import com.google.common.io.Closeables;
import com.xebialabs.deployit.plugin.generic.ci.Container;
import com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.RuntimeIOException;

import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;

@SuppressWarnings("serial")
public class TemplateArtifactCopyStep extends ArtifactCopyStep {

	private static final String MDC_KEY_TEMPLATE_PATH = "templatePath";

	private Map<String, Object> vars;

    private String templatePath;

    public TemplateArtifactCopyStep(int order, Container container, Map<String, Object> vars, String templatePath, String targetPath) {
        super(order, null, container, targetPath);
        this.vars = vars;
        this.templatePath = templatePath;
        Preconditions.checkNotNull(templatePath);
    }

    @Override
    protected Result doExecute() throws Exception {
    	MDC.put(MDC_KEY_TEMPLATE_PATH, templatePath);
    	try {
    		return super.doExecute();
    	} finally {
    		MDC.remove(MDC_KEY_TEMPLATE_PATH);
    	}
    }

    @Override
    protected OverthereFile resolveSourceFile() {
        OverthereFile localTemp = getLocalConnection().getTempFile("generic_plugin", ".tmp");
        evaluateTemplate(localTemp, templatePath, vars);
        return localTemp;
    }

    protected void evaluateTemplate(OverthereFile renderTo, String templatePath, Map<String, Object> vars) {
        Configuration cfg = ConfigurationHolder.getConfiguration();
        OutputStream out =  renderTo.getOutputStream();
        try {
            Template template = cfg.getTemplate(templatePath);
            template.process(vars, new OutputStreamWriter(out));

            if (logger.isDebugEnabled()) {
                InputStream in = renderTo.getInputStream();
                try {
                    logger.debug(new String(ByteStreams.toByteArray(in)));
                } finally {
                    Closeables.closeQuietly(in);
                }
            }

        } catch (IOException e) {
            throw new RuntimeIOException(e);
        } catch (TemplateException e) {
            throw new RuntimeException(e);
        } finally {
            Closeables.closeQuietly(out);
        }

    }
}
