package com.xebialabs.deployit.plugin.generic.step;

import com.google.common.base.Strings;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.generic.ci.Container;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Strings.nullToEmpty;

@SuppressWarnings("serial")
public class ArtifactDeleteStep extends GenericBaseStep {

    private DeployableArtifact artifact;
    private String targetDirectory;
    private boolean targetDirectoryShared;
    private String targetFile;
    private String description;

    public ArtifactDeleteStep(int order, Container container, DeployableArtifact artifact, String targetDirectory) {
        super(order, container);
        this.artifact = artifact;
        this.targetDirectory = checkNotNull(targetDirectory);
    }

    public ArtifactDeleteStep(int order, Container container, String targetDirectory) {
        this(order, container, null, targetDirectory);
    }

    @Override
    protected Result doExecute() throws Exception {
        if (artifact != null) {
            return deleteArtifact();
        } else {
            OverthereFile remoteFile = getRemoteConnection().getFile(targetDirectory);
            if (!nullToEmpty(targetFile).trim().isEmpty()) {
                remoteFile = remoteFile.getFile(targetFile);
            }
            deleteFile(remoteFile);
        }

        return Result.Success;
    }

    protected Result deleteArtifact() {
        OverthereFile localFile = artifact.getFile();
        OverthereFile remoteDir = getRemoteConnection().getFile(targetDirectory);

        if (localFile.isDirectory()) {
            if (targetDirectoryShared) {
                for (OverthereFile file : localFile.listFiles()) {
                    deleteFile(remoteDir.getFile(file.getName()));
                }
            } else {
                deleteFile(remoteDir);
            }
        } else {
            String name = nullToEmpty(targetFile).trim().isEmpty() ? artifact.getName() : targetFile;
            deleteFile(remoteDir.getFile(name));
        }

        return Result.Success;
    }

    protected void deleteFile(OverthereFile file) {
        getCtx().logOutput("Deleting " + file.getPath() + " on host " + getContainer().getHost());
        file.deleteRecursively();

    }

    @Override
    public String getDescription() {
        if (description == null) {
            return generateDescription();
        }
        return  description;
    }

    protected String generateDescription() {
        if (artifact != null) {
            return "Deleting " + artifact.getName() + " from " + getContainer().getHost();
        }

        if (!Strings.nullToEmpty(targetFile).trim().isEmpty()) {
                return "Deleting file " + targetFile + " from directory " + targetDirectory + " on host " +  getContainer().getHost();
        }

        return "Deleting directory " + targetDirectory + " from  " + getContainer().getHost();
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public String getTargetFile() {
        return targetFile;
    }

    public void setTargetFile(String targetFile) {
        this.targetFile = targetFile;
    }

    public boolean isTargetDirectoryShared() {
        return targetDirectoryShared;
    }

    public void setTargetDirectoryShared(boolean targetDirectoryShared) {
        this.targetDirectoryShared = targetDirectoryShared;
    }
}
