package com.xebialabs.deployit.plugin.ec2.access;

import java.io.Serializable;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeoutException;
import org.jclouds.ContextBuilder;
import org.jclouds.ec2.EC2AsyncClient;
import org.jclouds.ec2.EC2Client;
import org.jclouds.ec2.domain.InstanceState;
import org.jclouds.ec2.domain.Reservation;
import org.jclouds.ec2.domain.RunningInstance;
import org.jclouds.ec2.options.RunInstancesOptions;
import org.jclouds.rest.RestContext;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.plugin.cloud.util.Retrier;
import com.xebialabs.deployit.plugin.ec2.ci.Credentials;
import com.xebialabs.deployit.plugin.ec2.ci.HostTemplate;

import static com.google.common.collect.Iterables.getOnlyElement;
import static com.google.common.collect.Maps.newHashMap;

@SuppressWarnings("serial")
public class EC2Adapter implements Serializable {

    protected Credentials credentials;

    private int actionTimeout = 20; // seconds

    private int retryDelay = 5; // seconds

    public EC2Adapter(final Credentials credentials) {
        this.credentials = credentials;
    }

    /**
     * Alias with default timing parameters
     */
    public String kickNewInstance(final HostTemplate nodeTemplate, final String instanceLabel) throws TimeoutException {
        return kickNewInstance(nodeTemplate, instanceLabel, actionTimeout, retryDelay);
    }

    /**
     * Triggers new instance creation and gives back an ID.
     * Instance is not ready at this point yet.
     */
    public String kickNewInstance(final HostTemplate nodeTemplate, final String instanceLabel, int actionTimeout, int retryDelay) throws TimeoutException {

        final EC2Client client = client(credentials);

        Reservation<? extends RunningInstance> started = client.getInstanceServices().
                runInstancesInRegion(nodeTemplate.getRegion(), null, nodeTemplate.getAmi(), 1, 1,
                        RunInstancesOptions.Builder.
                                asType(nodeTemplate.getInstanceType().getValue()).
                                withKeyName(nodeTemplate.getKeyPair()).
                                withSecurityGroups(nodeTemplate.getGroups())
                );

        final RunningInstance instance = getOnlyElement(started);

        final Map<String, String> tags = newHashMap();
        tags.put("Name", instanceLabel);

        new Retrier<Boolean>(
                new Callable<Boolean>() {
                    @Override
                    public Boolean call() throws Exception {
                        client.getTagApiForRegion(nodeTemplate.getRegion()).get().applyToResources(tags, Lists.newArrayList(instance.getId()));
                        return true;
                    }
                }
        ).retryFor(actionTimeout, retryDelay);

        return instance.getId();
    }


    /**
     * Alias with default timing parameters
     */
    public String waitUntilInstanceIsRunning(final String region, final String instanceId) throws TimeoutException {
        return waitUntilInstanceIsRunning(region, instanceId, actionTimeout, retryDelay);
    }


    /**
     * Waits until instance becomes available according to the information retrieved from hypervisor
     * @return Public address of the running instance
     */
    public String waitUntilInstanceIsRunning(final String region, final String instanceId, final int actionTimeout, final int retryDelay) throws TimeoutException {
        final EC2Client client = client(credentials);

        return new Retrier<String>(new Callable<String>() {
            @Override
            public String call() throws Exception {
                RunningInstance instance = getOnlyElement(getOnlyElement(client.getInstanceServices().describeInstancesInRegion(region, instanceId)));
                if (InstanceState.RUNNING.equals(instance.getInstanceState())) {
                    return instance.getIpAddress();
                }

                throw new IllegalStateException("Expected state to be RUNNING, got " + instance.getInstanceState() + " instead");
            }
        }).retryFor(actionTimeout, retryDelay);
    }

    /**
     * Returns IP address of the instance
     * @return Public address of the running instance
     */
    public String getPublicIpAddress(final String region, final String instanceId) {
        return getOnlyElement(getOnlyElement(client(credentials).getInstanceServices().describeInstancesInRegion(region, instanceId))).getIpAddress();
    }


    /**
     * Destroys the instance
     */
    public void shutDown(final String region, final String cloudId) {
        client(credentials).getInstanceServices().terminateInstancesInRegion(region, cloudId);
    }


    // Helper

    private EC2Client client(Credentials credentials) {
        RestContext<EC2Client, EC2AsyncClient> context = ContextBuilder.newBuilder("aws-ec2").
                credentials(credentials.getKey(), credentials.getSecret()).
                build();

        return context.getApi();
    }
}
