package com.xebialabs.deployit.plugin.cmd.deployed;

import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.plugin.api.deployment.planning.*;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployed;
import com.xebialabs.deployit.plugin.cmd.ci.Command;
import com.xebialabs.deployit.plugin.cmd.step.ExecuteCommandStep;
import com.xebialabs.deployit.plugin.cmd.step.NoCommandStep;
import com.xebialabs.deployit.plugin.file.File;
import com.xebialabs.deployit.plugin.overthere.Host;

import java.util.List;
import java.util.Set;

import static com.google.common.collect.Sets.newHashSet;

@SuppressWarnings("serial")
@Metadata(description = "Command deployed to a Host")
public class DeployedCommand extends BaseDeployed<Command, Host> {

    private static final String UNDO_PREFIX = "/undo ";

    @Property(description = "Order of the command", defaultValue = "50")
    private int order = 50;

    @Property(required = false, description = "Artifacts that the command depends on")
    private Set<File> dependencies = newHashSet();

    @Property(required = true, size = Property.Size.LARGE, description = "Command line to execute on host. Dependent artifacts can be referred to using ${artifact name}.")
    private String commandLine;

    @Property(required = false, description = "Command to execute when undeploying command", category = "Undo")
    private Command undoCommand;

    @Property(required = false, size = Property.Size.LARGE, description = "Undo command line to execute on host. Dependent artifacts can be referred to using ${artifact name}.", category = "Undo")
    private String undoCommandLine;

    @Property(description = "Order of the undo command", defaultValue = "49", category = "Undo")
    private int undoOrder = 49;

    @Property(required = false, description = "Artifacts that the undo command depends on")
    private Set<File> undoDependencies = newHashSet();

    @Property(required = false, description = "Indicates whether the undoCommand should be run on an upgrade", category = "Undo")
    private boolean runUndoCommandOnUpgrade = true;

    @Property(required = false, isTransient = true, description = "Forces the command to be rerun.")
    private boolean rerunCommand;

    @Create
    public void executeCreateCommand(DeploymentPlanningContext ctx, Delta delta) {
        ctx.addStepWithCheckpoint(new ExecuteCommandStep(order, this), delta);
    }

    @Modify
    public void executeModifyCommand(DeploymentPlanningContext ctx, Delta delta) {
        setOrOverrideUndoCommand();
        if (undoCommand != null && runUndoCommandOnUpgrade) {
            DeployedCommand deployedUndoCommand = createDeployedUndoCommand();
            ctx.addStepWithCheckpoint(new ExecuteCommandStep(undoCommand.getOrder(), deployedUndoCommand), delta, Operation.DESTROY);
        }

        ctx.addStepWithCheckpoint(new ExecuteCommandStep(order, this), delta);
    }

    @Noop
    public void executeUpdateCommand(DeploymentPlanningContext ctx, Delta delta) {
        if (rerunCommand) {
            ctx.addStepWithCheckpoint(new ExecuteCommandStep(order, this), delta);
        }
    }

    private void setOrOverrideUndoCommand() {
        if (hasAttributeBasedUndoCommand()) {
            undoCommand = Type.valueOf(Command.class).getDescriptor().newInstance(this.getDeployable().getId());
            undoCommand.setCommandLine(undoCommandLine);
            undoCommand.setDependencies(undoDependencies);
            undoCommand.setOrder(undoOrder);
        }
    }

    private DeployedCommand createDeployedUndoCommand() {
        String id = getDeployedCommandId();
        DeployedCommand deployedUndoCommand = Type.valueOf(DeployedCommand.class).getDescriptor().newInstance(id);
        deployedUndoCommand.setDeployable(undoCommand);
        deployedUndoCommand.setContainer(getContainer());

        for (PropertyDescriptor property : undoCommand.getType().getDescriptor().getPropertyDescriptors()) {
            if (deployedUndoCommand.hasProperty(property.getName())) {
                deployedUndoCommand.setProperty(property.getName(), undoCommand.getProperty(property.getName()));
            }
        }

        return deployedUndoCommand;
    }

    @Destroy
    public void destroyCommand(DeploymentPlanningContext ctx, Delta delta) {
        setOrOverrideUndoCommand();
        if (undoCommand != null) {
            DeployedCommand deployedUndoCommand = createDeployedUndoCommand();
            ctx.addStepWithCheckpoint(new ExecuteCommandStep(undoCommand.getOrder(), deployedUndoCommand), delta);
        } else {
            ctx.addStepWithCheckpoint(new NoCommandStep(order, this), delta);
        }
    }

    public Set<File> getDependencies() {
        return dependencies;
    }

    public void setDependencies(Set<File> dependencies) {
        this.dependencies = dependencies;
    }

    public int getOrder() {
        return order;
    }

    public void setOrder(int order) {
        this.order = order;
    }

    public String getCommandLine() {
        return commandLine;
    }

    public void setCommandLine(String commandLine) {
        this.commandLine = commandLine;
    }

    public Command getUndoCommand() {
        return undoCommand;
    }

    public void setUndoCommand(Command undoCommand) {
        this.undoCommand = undoCommand;
    }

    public boolean isRunUndoCommandOnUpgrade() {
        return runUndoCommandOnUpgrade;
    }

    public void setRunUndoCommandOnUpgrade(boolean runUndoCommandOnUpgrade) {
        this.runUndoCommandOnUpgrade = runUndoCommandOnUpgrade;
    }

    public boolean isRerunCommand() {
        return rerunCommand;
    }

    public void setRerunCommand(boolean rerunCommand) {
        this.rerunCommand = rerunCommand;
    }

    public String getUndoCommandLine() {
        return undoCommandLine;
    }

    public void setUndoCommandLine(String undoCommandLine) {
        this.undoCommandLine = undoCommandLine;
    }

    public int getUndoOrder() {
        return undoOrder;
    }

    public void setUndoOrder(int undoOrder) {
        this.undoOrder = undoOrder;
    }

    public Set<File> getUndoDependencies() {
        return undoDependencies;
    }

    public void setUndoDependencies(Set<File> undoDependencies) {
        this.undoDependencies = undoDependencies;
    }

    private String getParentId(String id) {
        String[] nameParts = id.split("/");
        List<String> list = Lists.newArrayList(nameParts);
        if (list.size() > 1) {
            list.remove(nameParts.length - 1);
        }
        return Joiner.on("/").join(list);
    }

    private boolean hasAttributeBasedUndoCommand() {
        return undoCommandLine != null && !undoCommandLine.isEmpty();
    }

    private String getDeployedCommandId() {
        if (hasAttributeBasedUndoCommand()) {
            return getParentId(undoCommand.getId()) + UNDO_PREFIX + undoCommand.getName();
        }
        return undoCommand.getId();
    }
}
