package com.xebialabs.deployit.plugin.steps

import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, StepExitCode}
import com.xebialabs.deployit.plugin.api.rules.{RulePostConstruct, StepMetadata, StepParameter, StepPostConstructContext}
import com.xebialabs.deployit.plugin.overthere.{Host, HostContainer}
import com.xebialabs.xlplatform.satellite.{Satellite, SatelliteAware}

@StepMetadata(name = "wait")
class WaitStep(stepOrder: Integer, stepDescription: String, waitTimeInSeconds: Integer) extends SatelliteAware {

  def this() = this(null, null, null)

  def this(stepOrder: Integer, stepDescription: String, waitTimeInSeconds: Integer, host: Host) = {
    this(stepOrder, stepDescription, waitTimeInSeconds)
    container = host
  }

  private final val SECONDS_TO_SLEEP = 10

  @StepParameter(description = "Seconds that this step pause plan execution")
  private val seconds: Integer = waitTimeInSeconds

  @StepParameter(description = "Description of this step, as it should appear in generated deployment plans", calculated = true)
  private var description: String = stepDescription

  @StepParameter(description = "Execution order of the step")
  private val order: Integer = stepOrder

  private var container: HostContainer = _

  @RulePostConstruct
  private def doPostConstruct(ctx: StepPostConstructContext) {
    if (description == null || description.length() == 0) {
      description = s"Waiting for $seconds seconds"
    }
  }

  override def getOrder: Int = order

  override def getDescription: String = description

  override def execute(ctx: ExecutionContext): StepExitCode = {
    val sleepIntervals = seconds / SECONDS_TO_SLEEP
    val remainingSeconds = seconds % SECONDS_TO_SLEEP

    ctx.logOutput(getDescription)

    try {
      var secondsCountDown = seconds
      0 until sleepIntervals foreach { _ =>
        ctx.logOutput(s"Time remaining $secondsCountDown seconds.")
        Thread.sleep(SECONDS_TO_SLEEP * 1000)
        secondsCountDown -= SECONDS_TO_SLEEP
      }
      if (remainingSeconds > 0) {
        ctx.logOutput(s"Time remaining $remainingSeconds seconds.")
        Thread.sleep(remainingSeconds * 1000)
      }
      ctx.logOutput("Wait complete.")
    } catch {
      case i: InterruptedException =>
        ctx.logOutput("Wait interrupted.")
        Thread.currentThread().interrupt()
        return StepExitCode.FAIL
    }
    StepExitCode.SUCCESS
  }

  override def getSatellite: Satellite = {
    Option(container) match {
      case Some(_) => container.getHost.getSatellite
      case _ => null
    }
  }
}
