package com.xebialabs.deployit.plugin.steps

import java.util
import java.util.{Map => JMap}

import com.xebialabs.deployit.plugin.api.Deprecations
import com.xebialabs.deployit.plugin.api.flow._
import com.xebialabs.deployit.plugin.api.rules.{RulePostConstruct, StepMetadata, StepParameter, StepPostConstructContext}
import grizzled.slf4j.Logger
import com.xebialabs.platform.script.jython._
import com.xebialabs.platform.script.jython.ScriptSource.byResource

import scala.beans.BeanProperty

@StepMetadata(name = "jython")
class JythonStep extends BaseStep with PreviewStep {

  @transient private lazy val logger = Logger(getClass)

  @transient private lazy val jython = new JythonSupport{}

  @StepParameter(name="scriptPath", description = "DEPRECATED: Use 'script' instead. Path to the Python script to execute (relative to XL Deploy's classpath)", required = false)
  private val deprecatedScriptPath: String = ""

  @StepParameter(name="script", description = "Path to the Python script to execute (relative to XL Deploy's classpath)", required = false)
  @BeanProperty var scriptPath: String = ""

  @StepParameter(description = "Dictionary with keys as variable name and values as variable that are passed to the Python script", required = false, calculated = true)
  @BeanProperty var jythonContext: JMap[String, Any] = new util.HashMap[String, Any]()

  @RulePostConstruct
  def doPostConstruct(ctx: StepPostConstructContext): Unit = {
    if (isBlank(scriptPath)) {
      require(!isBlank(deprecatedScriptPath), "Cannot create step 'jython' since required parameter 'script' is not specified")
      Deprecations.deprecated("**Deprecated** 'script-path' parameter on a jython step is deprecated, use 'script' instead. (Will be removed in XL Deploy 5.5)")
      scriptPath = deprecatedScriptPath
    }

    calculateOrder(ctx)
    calculateDescription(ctx)
    jythonContext = ContextHelper.defaultContext(ctx, jythonContext)
  }

  private def isBlank(s:String) = {
    Option(s).map(_.trim.isEmpty).getOrElse(true)
  }

  override def execute(executionContext: ExecutionContext): StepExitCode = {
    try {
      executeScript(executionContext)
    } catch {
      case e: JythonException => handleError(executionContext, e)
    }
  }

  private def executeScript(executionContext: ExecutionContext): StepExitCode = {
    import scala.collection.convert.wrapAll._
    implicit val scriptContext = createJythonContext(executionContext, Bindings.xlDeployApiServices ++ (jythonContext + ("context" -> executionContext)).toMap )
    val script: AnyRef = jython.executeScript(byResource(scriptPath))
    StepExitCode.SUCCESS
  }

  private def createJythonContext(executionContext: ExecutionContext, variables: Map[String, Any]) = {
    JythonContext.withLibrariesAndFactory(Syntactic.loggerLib +: Syntactic.wrapperCodeWithLib(variables.keys)) {
      val scriptContext = variables.toScriptContext
      scriptContext.setWriter(new ConsumerWriter((text) => executionContext.logOutput(text)))
      scriptContext
    }
  }

  private def handleError(ctx: ExecutionContext, e: JythonException): StepExitCode = {
    logger.error(e.getMessage, e)
    ctx.logError(e.getMessage, e)
    StepExitCode.FAIL
  }

  override def getPreview: Preview = {
    Preview.withSourcePathAndContents(scriptPath, byResource(scriptPath).scriptContent)
  }
}
