package com.xebialabs.deployit.plugin.steps

import java.io.InputStream
import java.net.URL

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext
import com.xebialabs.overthere.OverthereFile
import com.xebialabs.overthere.util.OverthereUtils
import com.xebialabs.platform.sugar.ResourcesSugar

trait CopyFileSupport extends ResourcesSugar { self: TargetHostSupport =>

  private[steps] def mkTargetDirsIfRequired(targetFile: OverthereFile, createTargetPath: Boolean = true)(implicit ctx: ExecutionContext, targetHostEnvironment: HostEnvironment) = {
    val targetDirectory: OverthereFile = targetFile.getParentFile
    if (!targetDirectory.exists()) {
      if (createTargetPath) {
        ctx.logOutput(s"Creating path ${targetDirectory.getPath} on host $targetHost")
        targetDirectory.mkdirs()
      } else {
        ctx.logError(s"Path ${targetDirectory.getPath} on host $targetHost does not exist.")
        throw ExitStep
      }
    }
  }

  private[steps] def deleteTargetFileIfPresent(targetFile: OverthereFile)(implicit ctx: ExecutionContext, targetHostEnvironment: HostEnvironment) = if (targetFile.exists()) {
    ctx.logOutput(s"Deleting ${targetFile.getPath} since it already exists on host $targetHost")
    targetFile.deleteRecursively()
  }

  private[steps] def copyFileToTarget(sourceFile: OverthereFile, targetFile: OverthereFile)(implicit ctx: ExecutionContext, targetHostEnvironment: HostEnvironment): Unit = {
    ctx.logOutput(s"Copying ${sourceFile.getPath} to ${targetFile.getPath} on host $targetHost")
    sourceFile.copyTo(targetFile)
  }

  private[steps] def copyUrlToTarget(url: URL, targetFile: OverthereFile)(implicit ctx: ExecutionContext, targetHostEnvironment: HostEnvironment) {
    ctx.logOutput(s"Copying resource $url to ${targetFile.getPath} on host $targetHost")
    val inputStream: InputStream = url.openStream()
    try {
      OverthereUtils.write(inputStream, targetFile)
    } finally {
      inputStream.close()
    }
  }

  private[steps] case object ExitStep extends Exception
}
