package com.xebialabs.deployit.plugin.steps

import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, StepExitCode}
import com.xebialabs.deployit.plugin.api.rules.{Scope, StepParameter, StepPostConstructContext}
import com.xebialabs.deployit.plugin.api.udm.Container
import com.xebialabs.deployit.plugin.overthere.{DefaultExecutionOutputHandler, Host, HostContainer}
import com.xebialabs.deployit.plugin.steps.TargetContainerHelper.{defaultTargetContainer, targetByProperty}
import com.xebialabs.overthere.{CmdLine, OverthereConnection}
import com.xebialabs.platform._

import scala.reflect.ClassTag

object TargetContainerHelper {
  def defaultTargetContainer[T >: Container with Null](ctx: StepPostConstructContext)(implicit classTag: ClassTag[T]): T = {
    if (ctx.getScope == Scope.DEPLOYED && classTag.runtimeClass.isInstance(ctx.getDelta.correctDeployed.getContainer)) {
      ctx.getDelta.correctDeployed.getContainer.asInstanceOf[T]
    } else {
      null
    }
  }

  def targetByProperty[T >: Container with Null : ClassTag](ctx: StepPostConstructContext, property: String): T = {
    if (ctx.getScope == Scope.DEPLOYED) {
      val container: Container = ctx.getDelta.correctDeployed.getContainer.asInstanceOf[Container]
      if (container.hasProperty(property) && implicitly[ClassTag[T]].runtimeClass.isInstance(container.getProperty(property))) {
        return container.getProperty[T](property)
      }
    }
    null
  }
}

object TargetContainerJavaHelper {
  def defaultTargetContainer[T >: Container with Null](ctx: StepPostConstructContext, containerClass: Class[T]): T = {
    TargetContainerHelper.defaultTargetContainer(ctx)(ClassTag(containerClass))
  }
}

trait TargetHostSupport {

  @StepParameter(description = "A target host where the step is applied", calculated = true)
  protected[steps] var targetHost: Host = _

  def execute(ctx: ExecutionContext): StepExitCode = {
    implicit val hostEnvironment = HostEnvironment(targetHost)
    try {
      executeOnTargetHost(ctx, hostEnvironment)
    } finally {
      hostEnvironment.connection.close()
    }
  }

  protected[steps] def executeOnTargetHost(implicit ctx: ExecutionContext, targetHostEnvironment: HostEnvironment): StepExitCode

  protected[steps] def calculateTargetHost(ctx: StepPostConstructContext) = {
    if (targetHost == null) {
      targetHost = Option(defaultTargetContainer[HostContainer](ctx)).map(_.getHost).getOrElse(targetByProperty[Host](ctx, "host"))
    }
  }

  protected[steps] def targetOs = targetHost.getOs

  protected[steps] def remoteFile(path: String)(implicit targetHostEnvironment: HostEnvironment) = targetHostEnvironment.remoteFile(path)

  protected[steps] def remoteTempFile(fileName: String)(implicit targetHostEnvironment: HostEnvironment) = targetHostEnvironment.remoteTempFile(fileName)

  protected[steps] def fileInWorkDir(fileName: String)(implicit targetHostEnvironment: HostEnvironment) = targetHostEnvironment.workingDir.getFile(fileName)

  protected[steps] def executeCommand(command: CmdLine)(implicit executionContext: ExecutionContext, targetHostEnvironment: HostEnvironment): StepExitCode = {
    val stdoutHandler = DefaultExecutionOutputHandler.handleStdout(executionContext)
    val stderrHandler = DefaultExecutionOutputHandler.handleStderr(executionContext)
    try {
      executionContext.logOutput(s"Executing ${command.toCommandLine(targetOs, true)} on host $targetHost")
      val exitCode: Int = targetHostEnvironment.connection.execute(stdoutHandler, stderrHandler, command)
      if (exitCode != 0) {
        executionContext.logError(s"Execution failed with exit code $exitCode")
        StepExitCode.FAIL
      } else {
        StepExitCode.SUCCESS
      }
    } finally {
      stderrHandler.close()
      stdoutHandler.close()
    }
  }
}

object HostEnvironment {
  def apply(host: Host) : HostEnvironment = {
    HostEnvironment(Option(host).map(_.getConnection).getOrElse(throw new IllegalStateException("targetHost is not set")))
  }
} 

case class HostEnvironment(connection: OverthereConnection) {
  lazy val workingDir = {
    val tmpDir = remoteTempFile("working.directory.tmp")
    tmpDir.mkdir()
    connection.setWorkingDirectory(tmpDir)
    tmpDir
  }

  protected[steps] def remoteTempFile(fileName: String) = {
    connection.getTempFile(fileName)
  }

  protected[steps] def remoteFile(path: String) = connection.getFile(path)
}