package com.xebialabs.deployit.plugin.steps

import java.util
import java.util.{List => JList}

import com.xebialabs.deployit.plugin.api.flow._
import com.xebialabs.deployit.plugin.api.rules.{RulePostConstruct, StepMetadata, StepParameter, StepPostConstructContext}
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact
import com.xebialabs.deployit.plugin.remoting.preview.PreviewOverthereConnection
import com.xebialabs.overthere.util.OverthereUtils
import com.xebialabs.overthere.{CmdLine, OperatingSystemFamily, OverthereFile}

import scala.collection.convert.wrapAsScala._
import scala.io.Source.fromInputStream

@StepMetadata(name = "os-script")
class OsScriptStep(context: util.Map[String, Any] = new util.HashMap[String, Any]())
  extends BaseStep with TargetHostSupport with FreemarkerSupport with CopyFileSupport with PreviewStep with StageableStep {

  def this() = this(new util.HashMap[String, Any]())

  freemarkerContext = context

  @StepParameter(name = "script", description = "Script base name that will be executed on the target host")
  protected[steps] val scriptBaseName: String = null

  @StepParameter(description = "Resources that are attached to the script and uploaded to the target host", required = false)
  protected[steps] val classpathResources: JList[String] = new util.ArrayList[String]

  @StepParameter(name = "uploadArtifacts", description = "If true, every artifact presented in the freemarker context will be uploaded to the working directory of the script", required = false)
  protected[steps] val uploadArtifactsInFreemarkerContext: Boolean = true

  private var stagedArtifacts: Vector[StagedArtifact] = Vector()

  @RulePostConstruct
  def doPostConstruct(ctx: StepPostConstructContext): Unit = {
    calculateOrder(ctx)
    calculateDescription(ctx)
    calculateTargetHost(ctx)
    calculateFreemarkerContext(ctx)
  }

  override def requestStaging(ctx: StagingContext): Unit = stagedArtifacts = uploadArtifactsInFreemarkerContext match {
    case false => Vector.empty // Nothing to do, please go on.
    case _ => freemarkerContext.toList.collect { case (_, a: Artifact) => StagedArtifact(ctx.stageArtifact(a, targetHost), a) }.toVector
  }

  override protected[steps] def executeOnTargetHost(implicit ctx: ExecutionContext, targetHostEnvironment: HostEnvironment): StepExitCode = {
    val artifactFiles = uploadArtifacts
    val targetScriptName = processScript(artifactFiles)
    processClasspathResources(artifactFiles)
    executeScript(targetScriptName)
  }

  private[steps] def processScript(artifactFiles: Map[ArtifactKey, OverthereFile])(implicit executionContext: ExecutionContext, targetHostEnvironment: HostEnvironment) = {
    val resolvedScriptResource = resolveScriptResource
    processAndUploadResource(resolvedScriptResource, artifactFiles)
    resolvedScriptResource.stripSuffix(".ftl")
  }

  private[steps] def processClasspathResources(artifactFiles: Map[ArtifactKey, OverthereFile])(implicit executionContext: ExecutionContext, targetHostEnvironment: HostEnvironment) = {
    Option(classpathResources).getOrElse(new util.ArrayList[String]()) foreach { resource => processAndUploadResource(resource, artifactFiles) }
  }

  private[steps] def processAndUploadResource(resourceName: String, artifactFiles: Map[ArtifactKey, OverthereFile])(implicit executionContext: ExecutionContext, targetHostEnvironment: HostEnvironment) = {
    val targetFile = fileInWorkDir(resourceName.stripSuffix(".ftl"))

    mkTargetDirsIfRequired(targetFile)

    if (resourceName.endsWith(".ftl")) {
      renderFreemarkerTo(resourceName, targetFile, artifactFiles)
    } else {
      copyUrlToTarget(resource(resourceName), targetFile)
    }
  }


  private[steps] def uploadArtifacts(implicit executionContext: ExecutionContext, targetHostEnvironment: HostEnvironment): Map[ArtifactKey, OverthereFile] = {
    stagedArtifacts.map { sa =>
      val remoteFile: OverthereFile = OverthereUtils.getUniqueFolder(targetHostEnvironment.workingDir, "xld").getFile(sa.artifact.getFile.getName)
      copyFileToTarget(sa.stagedFile.get(targetHostEnvironment.connection, executionContext), remoteFile)
      (ArtifactKey(sa.artifact), remoteFile)
    }.toMap
  }

  private[steps] def executeScript(scriptResource: String)(implicit ctx: ExecutionContext, targetHostEnvironment: HostEnvironment): StepExitCode = {
    val executable: OverthereFile = fileInWorkDir(scriptResource)
    executable.setExecutable(true)
    executeCommand(CmdLine.build(executable.getPath))
  }

  private[steps] def resolveScriptResource = {
    val possibleScriptNames = (targetOs match {
      case OperatingSystemFamily.WINDOWS => List(".bat.ftl", ".cmd.ftl", ".bat", ".cmd", "")
      case _ => List(".sh.ftl", ".sh", "")
    }).map(scriptBaseName + _)

    val resolvedScriptName = possibleScriptNames.find(resources(_).nonEmpty)
    require(resolvedScriptName.isDefined, s"No candidates found for script basename $scriptBaseName")
    resolvedScriptName.get
  }

  override lazy val getPreview: Preview = {
    val artifactFiles = stagedArtifacts.map { sa =>
      val previewConnection = PreviewOverthereConnection.getPreviewConnection
      val tempDir = previewConnection.getTempFile("preview", null)
      val remoteFile = previewConnection.getFile(tempDir, sa.artifact.getFile.getName)
      (ArtifactKey(sa.artifact), remoteFile)
    }.toMap
    val scriptPath = resolveScriptResource
    val scriptContent = if (scriptPath.endsWith(".ftl"))
      processWithFreemarker(scriptPath, maskPassword = true, artifactFiles)
    else
      fromInputStream(resourceAsInputStream(scriptPath)).getLines().mkString("\n")
    Preview.withSourcePathAndContents(scriptPath, scriptContent)
  }

  private case class StagedArtifact(stagedFile: StagedFile, artifact: Artifact) extends Serializable
}
