package com.xebialabs.deployit.plugin.steps

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.flow.StepExitCode
import com.xebialabs.deployit.plugin.api.rules.RulePostConstruct
import com.xebialabs.deployit.plugin.api.rules.StepMetadata
import com.xebialabs.deployit.plugin.api.rules.StepParameter
import com.xebialabs.deployit.plugin.api.rules.StepPostConstructContext

@StepMetadata(name = "wait")
class WaitStep(stepOrder: Integer, stepDescription: String, waitTimeInSeconds: Integer) extends Step {

  def this() = this(null, null, null)

  private final val SECONDS_TO_SLEEP = 10

  @StepParameter(description = "Seconds that this step pause plan execution")
  private val seconds: Integer = waitTimeInSeconds

  @StepParameter(description = "Description of this step, as it should appear in generated deployment plans", calculated = true)
  private var description: String = stepDescription

  @StepParameter(description = "Execution order of the step")
  private val order: Integer = stepOrder

  @RulePostConstruct
  private def doPostConstruct(ctx: StepPostConstructContext) {
    if (description == null || description.length() == 0) {
      description = s"Waiting for $seconds seconds"
    }
  }

  override def getOrder: Int = order

  override def getDescription: String = description

  override def execute(ctx: ExecutionContext): StepExitCode = {
    val sleepIntervals = seconds / SECONDS_TO_SLEEP
    val remainingSeconds = seconds % SECONDS_TO_SLEEP

    ctx.logOutput(getDescription)

    try {
      var secondsCountDown = seconds
      0 until sleepIntervals foreach { _ =>
        ctx.logOutput(s"Time remaining $secondsCountDown seconds.")
        Thread.sleep(SECONDS_TO_SLEEP * 1000)
        secondsCountDown -= SECONDS_TO_SLEEP
      }
      if (remainingSeconds > 0) {
        ctx.logOutput(s"Time remaining $remainingSeconds seconds.")
        Thread.sleep(remainingSeconds * 1000)
      }
      ctx.logOutput("Wait complete.")
    } catch {
      case i: InterruptedException =>
        ctx.logOutput("Wait interrupted.")
        Thread.currentThread().interrupt()
        return StepExitCode.FAIL
    }
    StepExitCode.SUCCESS
  }
}
