package com.xebialabs.deployit.plugin.dictionary;

import com.google.common.collect.Maps;
import com.xebialabs.deployit.ci.Application;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.Environment;
import com.xebialabs.deployit.ci.mapping.KeyValuePair;
import com.xebialabs.deployit.plugin.dictionary.ci.Dictionary;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * process List of key value pair based on dictionaries configured in the target environment
 */
public class AutoFiller {

	protected final Deployment deployment;

	public AutoFiller(Deployment deployment) {
		this.deployment = deployment;
	}

	public void process(List<KeyValuePair> keyValuePairs) {
		final Application application = deployment.getSource().getApplication();
		final Environment environment = deployment.getTarget();

		logger.debug("input kvp " + keyValuePairs);

		final Map<String, String> flattenKeyValuePairs = getAvailableDictionary(deployment);
		if (logger.isDebugEnabled()) {
			logger.debug("available key value pair: " + flattenKeyValuePairs);
		}
		for (KeyValuePair actualEntry : keyValuePairs) {
			if (StringUtils.isNotBlank(actualEntry.getValue())) {
				if (logger.isDebugEnabled())
					logger.debug("already overridden property for key " + actualEntry.getKey() + " -> " + actualEntry.getValue());

				continue;
			}

			final String value = findMatchingValue(actualEntry.getKey(), flattenKeyValuePairs, application, environment);
			if (value != null) {
				if (logger.isDebugEnabled())
					logger.debug("resolved property for key " + actualEntry.getKey() + " -> " + value);
				actualEntry.setValue(value);
			}
		}
	}

	private String findMatchingValue(String key, Map<String, String> flattenKeyValuePairs, Application application, Environment environment) {

		//env.key
		final String envName = StringUtils.substringAfterLast(environment.getLabel(), "/");
		final String envKey = envName.toLowerCase() + "." + key;
		logger.debug("envKey " + envKey);

		//app.key
		final String appName = StringUtils.substringAfterLast(application.getLabel(), "/");
		final String appKey = appName.toLowerCase() + "." + key;
		logger.debug("appKey " + appKey);

		//app.env.key
		final String appEnvKey = appName.toLowerCase() + "." + envName.toLowerCase() + "." + key;
		logger.debug("appEnvKey " + appEnvKey);

		if (flattenKeyValuePairs.containsKey(appEnvKey))
			return flattenKeyValuePairs.get(appEnvKey);

		if (flattenKeyValuePairs.containsKey(envKey))
			return flattenKeyValuePairs.get(envKey);

		if (flattenKeyValuePairs.containsKey(appKey))
			return flattenKeyValuePairs.get(appKey);

		//key without prefix
		if (flattenKeyValuePairs.containsKey(key))
			return flattenKeyValuePairs.get(key);

		return null;
	}

	protected Map<String, String> getAvailableDictionary(Deployment deployment) {
		Map<String, String> result = Maps.newHashMap();
		final Environment env = deployment.getTarget();
		final Set<Dictionary> membersOfType = env.getMembersOfType(Dictionary.class);
		for (Dictionary d : membersOfType) {
			result.putAll(KeyValuePair.toMap(d.getKeyValuePairs()));
		}

		return result;
	}

	private static Logger logger = Logger.getLogger(AutoFiller.class);
}
