package com.xebialabs.deployit.plugin.dictionary;

import com.google.common.base.Predicate;
import com.google.common.collect.Maps;
import com.xebialabs.deployit.ci.Application;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.mapping.KeyValuePair;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * As the importer does not scan Artifacts, first we need to Search in all the dictionaries
 * all the placeholders of the archive using this syntax:
 * <ul>
 * <li>placeholder.type.xxx.yyy.zzz=app1,app2,app3</li>
 * <li>placeholder.type.xxx.yyy.zzz=*</li>
 * <ul>
 * with ${type} could be ear, datasource, war (in lower case)....
 * then call the AutoFiller process operations.
 */
public class ArtifactAutoFiller extends AutoFiller {

	public static final String PREFIX = "placeholder";
	public static final String ALL_APPS = "*";

	private final String prefix;

	public ArtifactAutoFiller(Deployment deployment, String type) {
		super(deployment);
		prefix = String.format("%s.%s.", PREFIX, type);

	}

	@Override
	public void process(List<KeyValuePair> keyValuePairs) {
		if (!keyValuePairs.isEmpty()) {
			logger.debug("key value pair are not empty " + keyValuePairs + ", return");
		}

		//assert keyValuePairs.isEmpty()
		final Map<String, String> flattenKeyValuePairs = getAvailableDictionary(deployment);
		if (logger.isDebugEnabled()) {
			logger.debug("available key value pair: " + flattenKeyValuePairs);
		}

		for (String key : getKeys(flattenKeyValuePairs, deployment.getSource().getApplication())) {
			keyValuePairs.add(new KeyValuePair(key.substring(prefix.length()), null));
		}

		super.process(keyValuePairs);

	}

	private Set<String> getKeys(Map<String, String> flattenKeyValuePairs, final Application application) {
		final String appName = StringUtils.substringAfterLast(application.getLabel(), "/");
		return Maps.filterEntries(flattenKeyValuePairs, new Predicate<Map.Entry<String, String>>() {
			public boolean apply(final Map.Entry<String, String> input) {
				return isKey(input) && isValue(input, appName);
			}

			private boolean isKey(final Map.Entry<String, String> input) {
				return input.getKey().startsWith(prefix);
			}

			private boolean isValue(final Map.Entry<String, String> input, String appName) {
				final String value = input.getValue();
				if (StringUtils.isBlank(value))
					return false;

				if (value.equals(ALL_APPS))
					return true;

				for (String split : StringUtils.split(value, ",")) {
					if (appName.equals(split))
						return true;
				}
				return false;
			}
		}).keySet();
	}


	private static Logger logger = Logger.getLogger(ArtifactAutoFiller.class);
}
