package com.xebialabs.deployit.steps;

import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.StepExecutionContext;
import com.xebialabs.deployit.StepExecutionContextCallbackHandler;
import com.xebialabs.deployit.ci.Database;
import com.xebialabs.deployit.ci.artifact.SqlFolder;
import com.xebialabs.deployit.ci.artifact.mapping.PlaceholderFormat;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.hostsession.HostFile;
import com.xebialabs.deployit.hostsession.HostSession;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;


public class RunSqlScriptWithUploadedScriptOnDatabaseStep extends AbstractDatabaseStep implements Step {
	private final File sqlFile;
	private final String relativeSqlFile;

	public RunSqlScriptWithUploadedScriptOnDatabaseStep(Database target, SqlFolder source, File sqlFile, Map<String, String> properties, PlaceholderFormat placeholderFormat) {
		super(source, target,properties, placeholderFormat);
		this.sqlFile = sqlFile;
		this.relativeSqlFile = getRelativePath();
	}

	private String getRelativePath() {
		//location + relativePath = sqlFilePath
		return sqlFile.getAbsolutePath().substring(new File(source.getLocation()).getAbsolutePath().length() + 1);
	}

	public String getDescription() {
		return "Executing SQL Script " + this.sqlFile + " on " + database.getLabel();
	}

	public boolean execute(StepExecutionContext ctx) {
		HostSession s = getHostSession(ctx);
		try {
			final HostFile remoteSqlFile = getRemoteLocation(ctx).getFile(relativeSqlFile) ;

			final SqlErrorContextCallbackHandler handler = new SqlErrorContextCallbackHandler(ctx);
			int retCode = s.execute(handler, getSqlRunnerPath(ctx), remoteSqlFile.getPath());

			if (handler.hasErrors())
				throw new RuntimeException("SQL errors has been detected when running " + remoteSqlFile + ":" + handler.getErrors());

			return retCode == 0;

		} catch (RuntimeIOException exc) {
			throw new RuntimeIOException("Cannot execute remote command \"" + database.getCommand() + "\" on " + database.getHost(), exc);
		}
	}

	
	private class SqlErrorContextCallbackHandler extends StepExecutionContextCallbackHandler {

		private final List<String> errors = new ArrayList<String>();

		public SqlErrorContextCallbackHandler(StepExecutionContext ctx) {
			super(ctx);
		}

		@Override
		public void handleOutputLine(String line) {
			super.handleOutputLine(line);
			errorHandler(line);
		}


		@Override
		public void handleErrorLine(String line) {
			super.handleErrorLine(line);
			errorHandler(line);
		}

		private void errorHandler(String line) {
			if (line.startsWith("ERROR"))
				errors.add(line);
		}

		public boolean hasErrors() {
			return !errors.isEmpty();
		}

		public String getErrors() {
			return errors.toString();
		}
	}
}
