/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.ci;

import java.io.Serializable;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Predicate;

import com.xebialabs.deployit.BaseConfigurationItem;
import com.xebialabs.deployit.ConfigurationItem;
import com.xebialabs.deployit.ConfigurationItemProperty;
import com.xebialabs.deployit.ci.mapping.Mapping;

/**
 * A deployment of a {@link DeploymentPackage} on an {@link Environment}.
 */
@SuppressWarnings("serial")
@ConfigurationItem(category = "deployments", description = "A deployment of an application package or a deployable artifact to a middleware CI or an environment.")
public class Deployment extends BaseConfigurationItem {

	@ConfigurationItemProperty(required = false, label="Default virtual host", description = "The virtual host <hostname>:<portnumber> as used by the HTTP and application server. Valid examples; www.xebialabs.com:80")
	private String vhostDefinition = null;

	@ConfigurationItemProperty(required = true)
	private DeploymentPackage source;

	@ConfigurationItemProperty(required = true, asContainment = true)
	private Environment target;

	@SuppressWarnings("rawtypes")
    @ConfigurationItemProperty
	private Set<Mapping> mappings = new HashSet<Mapping>();

	public Deployment() {
		//	
	}

	public Deployment(DeploymentPackage source, Environment target) {
		this.source = source;
		this.target = target;
		this.label = "Deployment of " + source + " on " + target;
	}

	public Deployment(DeploymentPackage source, Environment target, String virtualHost) {
		this(source, target);
		this.vhostDefinition = virtualHost;
	}

	public String getVhostDefinition() {
		return vhostDefinition;
	}

	public void setVhostDefinition(String vhostDefinition) {
		this.vhostDefinition = vhostDefinition;
	}

	public DeploymentPackage getSource() {
		return source;
	}

	public void setSource(DeploymentPackage source) {
		this.source = source;
	}

	public Environment getTarget() {
		return target;
	}

	public void setTarget(Environment target) {
		this.target = target;
	}

	@SuppressWarnings("rawtypes")
    public Set<Mapping> getMappings() {
		return mappings;
	}

	/**
	 * Returns the mappings on this CI that are of a certain classes (or one of its subclasses) and have a certain
	 * source.
	 * 
	 * @param <M>
	 *            the type of the mappings sought after
	 * @param mappingClass
	 *            the class of the mappings sought after
	 * @param source
	 *            the source of the mappings sought after
	 * @return a list of mappings that are of class <tt>mappingClass</tt> or one of its subclasses and that have
	 *         <tt>source</tt> as their source.
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	public <M extends Mapping> List<M> getMappingsForSource(final Class<M> mappingClass, final Serializable source) {
		return (List<M>) CollectionUtils.select(mappings, new Predicate() {
			public boolean evaluate(Object input) {
				if (!mappingClass.isInstance(input)) {
					return false;
				} else {
					Mapping eachMapping = (Mapping) input;
					return eachMapping.getSource().equals(source);
				}
			}
		});
	}

	/**
	 * Returns the mappings on this CI that are of a certain classes (or one of its subclasses) and have a certain
	 * target.
	 * 
	 * @param <M>
	 *            the type of the mappings sought after
	 * @param mappingClass
	 *            the class of the mappings sought after
	 * @param target
	 *            the target of the mappings sought after
	 * @return a list of mappings that are of class <tt>mappingClass</tt> or one of its subclasses and that have
	 *         <tt>target</tt> as their target.
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	public <M extends Mapping> List<M> getMappingsForTarget(final Class<M> mappingClass, final Serializable target) {
		return (List<M>) CollectionUtils.select(mappings, new Predicate() {
			public boolean evaluate(Object input) {
				if (!mappingClass.isInstance(input)) {
					return false;
				} else {
					Mapping eachMapping = (Mapping) input;
					return eachMapping.getTarget().equals(target);
				}
			}
		});
	}

	/**
	 * Returns the mappings on this CI that are of a certain classes (or one of its subclasses).
	 * 
	 * @param <M>
	 *            the type of the mappings sought after
	 * @param mappingClass
	 *            the class of the mappings sought after
	 * @return a list of mappings that are of class <tt>mappingClass</tt> or one of its subclasses.
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	public <M extends Mapping> List<M> getMappingsOfType(final Class<M> mappingClass) {
		return (List<M>) CollectionUtils.select(mappings, new Predicate() {
			public boolean evaluate(Object input) {
				return mappingClass.isInstance(input);
			}
		});
	}

	/**
	 * Sets the mappings on this deployment.
	 * 
	 * @param mappings
	 *            the mappings to set.
	 */
	@SuppressWarnings("rawtypes")
    public void setMappings(Set<Mapping> mappings) {
		this.mappings = mappings;
	}

	/**
	 * Adds a mapping to this deployment.
	 * 
	 * @param mapping
	 *            the mapping to add.
	 */
	@SuppressWarnings("rawtypes")
    public void addMapping(Mapping mapping) {
		this.mappings.add(mapping);
	}

}
