/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.steps;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.FileSystemResourceLoader;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;

import com.xebialabs.deployit.StepExecutionContext;
import com.xebialabs.deployit.StepExecutionContextCallbackHandler;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.hostsession.CommandExecutionCallbackHandler;
import com.xebialabs.deployit.hostsession.HostFile;
import com.xebialabs.deployit.hostsession.HostFileUtils;
import com.xebialabs.deployit.hostsession.HostSession;
import com.xebialabs.deployit.hostsession.HostSessionFactory;
import com.xebialabs.deployit.mapper.Pair;
import com.xebialabs.deployit.util.TemplateResolver;

@SuppressWarnings("serial")
public class HostCommandExecutionStep extends HostSessionStep {

	private static final String CLASSPATH_RESOURCE_PROTOCOL = "classpath";

	private static final String FILE_RESOURCE_PROTOCOL = "file";

	private static final String LITERAL_RESOURCE_PROTOCOL = "literal";

	private static final String TEXT_RESOURCE_PROTOCOL = "text";

	private static final String ZIP_RESOURCE_PROTOCOL = "zip";

	public static final String CLASSPATH_RESOURCE_PREFIX = CLASSPATH_RESOURCE_PROTOCOL + ":";

	public static final String FILE_RESOURCE_PREFIX = FILE_RESOURCE_PROTOCOL + ":";

	public static final String LITERAL_RESOURCE_PREFIX = LITERAL_RESOURCE_PROTOCOL + ":";

	public static final String TEXT_RESOURCE_PREFIX = TEXT_RESOURCE_PROTOCOL + ":";

	public static final String ZIP_RESOURCE_PREFIX = ZIP_RESOURCE_PROTOCOL + ":";

	public static final String EXECUTING_HOST_COMMAND_DESCRIPTION_PREFIX = "";
	
	private static final Host LOCALHOST = Host.getLocalHost(); 

	private String[] command;

	private int expectedResult;

	private Map<String, String> inputResponse;

	private Map<String, String> resourcesToUpload;
	
	public HostCommandExecutionStep(Host host, String description, int expectedResult, String... command) {
		this(host, description, expectedResult, new int[0], command);
	}	
	
	public HostCommandExecutionStep(String description, int[] nondisclosableArgIndices, String... command) {
		this(LOCALHOST, description, 0, nondisclosableArgIndices, command);
	}

	// if non-empty, the indicies of the array of non-disclosable arguments indicates which of the vararg parameters should be hidden 
	public HostCommandExecutionStep(Host host, String description, int expectedResult, int[] nondisclosableArgIndices, String... command) {
		super(host);
		this.command = command;
		this.expectedResult = expectedResult;
		this.inputResponse = new HashMap<String, String>();
		this.resourcesToUpload = new HashMap<String, String>();
		setDescription(description);

		if (logger.isDebugEnabled()) {
			printDebugCreationMessage(host, nondisclosableArgIndices, command);
		}
		
	}
	
	private static void printDebugCreationMessage(Host host, int[] nondisclosableArgIndices, String[] command) {
		
		if (nondisclosableArgIndices == null) {
			throw new IllegalArgumentException("'nondisclosableArgIndices' may not be null");
		}
	
		String[] commandCopy = command.clone();
		
		for (int i = 0; i < nondisclosableArgIndices.length; i++) {
			assert ((nondisclosableArgIndices[i] >= 0) && (nondisclosableArgIndices[i] < command.length))
			: new Object[] { nondisclosableArgIndices, command };
			
			if (commandCopy[nondisclosableArgIndices[i]] != null) {
				// replace the argument by a string of ***
				commandCopy[nondisclosableArgIndices[i]] = 
					StringUtils.repeat("*", commandCopy[nondisclosableArgIndices[i]].length());
			} else {
				commandCopy[nondisclosableArgIndices[i]] = "****";
			}
		}
		
		logger.debug("Created HostCommandExecutionStep for host " + host.getLabel() + ":" + StringUtils.join(commandCopy, " "));
	}

	public void setExpectedResult(int expectedResult) {
		this.expectedResult = expectedResult;
	}

	public void addInputResponse(String input, String response) {
		inputResponse.put(input, response);
	}

	public void addResourceToUpload(String key, String resourceSpec) {
		resourcesToUpload.put(key, resourceSpec);
	}

	public boolean execute(StepExecutionContext ctx) {
		HostSession s = getHostSession();
		try {
			Map<String, String> commandVariables = new HashMap<String, String>();
			uploadResources(s, commandVariables);
			int res = executeCommand(s, commandVariables, ctx);

			return (res == expectedResult);
		} finally {
			s.close();
		}
	}

	private void uploadResources(HostSession s, Map<String, String> commandVariables) {
		for (Map.Entry<String, String> each : resourcesToUpload.entrySet()) {
			String key = each.getKey();
			String resourceSpec = each.getValue();
			HostFile hostFile = uploadResource(s, key, resourceSpec);
			commandVariables.put(key, hostFile.getPath());
		}
	}

	private HostFile uploadResource(HostSession s, String key, String resourceSpec) {
		Pair<String, String> parsedResourceSpec = parseResourceSpec(resourceSpec);
		String protocol = parsedResourceSpec.getFirst();
		if (protocol.equals(TEXT_RESOURCE_PROTOCOL)) {
			return uploadTextResource(s, resourceSpec);
		} else if (protocol.equals(ZIP_RESOURCE_PROTOCOL)) {
			throw new RuntimeException("can't handle zips in this step");
			// return uploadZipResource(s, resourceSpec);
		} else {
			return uploadSimpleResource(s, resourceSpec);
		}
	}

	private HostFile uploadSimpleResource(HostSession s, String resourceSpec) {
		Resource resource = getResource(resourceSpec);
		HostFile hostFile = s.getTempFile(getResourceName(resource));
		try {
			File resourceFile = resource.getFile();
			uploadFileResource(hostFile, resourceFile, resourceSpec);
		} catch (IOException exc) {
			uploadNonFileResource(hostFile, resource, resourceSpec);
		}
		return hostFile;

	}

	private void uploadFileResource(HostFile hostFile, File localFile, String resourceSpec) {
		if (localFile.isDirectory()) {
			uploadDirectory(hostFile, localFile, resourceSpec);
		} else {
			uploadRegularFile(hostFile, localFile, resourceSpec);
		}
	}

	private void uploadDirectory(HostFile hostFile, File localDir, String resourceSpec) {
		HostSession localSession = HostSessionFactory.getHostSession(Host.getLocalHost());
		try {
			HostFileUtils.copyDirectory(localSession.getFile(localDir.getPath()), hostFile);
		} finally {
			localSession.close();
		}
	}

	private void uploadRegularFile(HostFile hostFile, File localFile, String resourceSpec) {
		hostFile.put(localFile);
	}

	private void uploadNonFileResource(HostFile hostFile, Resource resource, String resourceSpec) {
		try {
			InputStream resourceIn = resource.getInputStream();
			try {
				byte[] resourceBytes = IOUtils.toByteArray(resourceIn);
				hostFile.put(new ByteArrayInputStream(resourceBytes), resourceBytes.length);
			} finally {
				resourceIn.close();
			}
		} catch (IOException exc) {
			throw new RuntimeIOException("Cannot upload non-file resource " + resourceSpec, exc);
		}
	}

	private HostFile uploadTextResource(HostSession s, String resourceSpec) {
		try {
			Pair<String, String> parsedResource = parseResourceSpec(resourceSpec);
			assert (parsedResource.getFirst().equals(TEXT_RESOURCE_PROTOCOL));
			String textResourceSpec = parsedResource.getSecond();

			Resource textResource = getResource(textResourceSpec);
			String unconvertedText = IOUtils.toString(textResource.getInputStream());
			String convertedText = s.getHostOperatingSystem().convertText(unconvertedText);
			byte[] convertedBytes = convertedText.getBytes("UTF-8");
			HostFile hostFile = s.getTempFile(getResourceName(textResource));
			hostFile.put(new ByteArrayInputStream(convertedBytes), convertedBytes.length);
			return hostFile;
		} catch (IOException exc) {
			throw new RuntimeIOException("Cannot upload text resource " + resourceSpec, exc);
		}
	}

	private Resource getResource(String resourceSpec) {
		Pair<String, String> parsedResourceSpec = parseResourceSpec(resourceSpec);
		String protocol = parsedResourceSpec.getFirst();
		String path = parsedResourceSpec.getSecond();

		if (protocol.equals(LITERAL_RESOURCE_PROTOCOL)) {
			try {
				return new ByteArrayResource(path.getBytes("UTF-8"));
			} catch (UnsupportedEncodingException exc) {
				return new ByteArrayResource(path.getBytes());
			}
		} else {
			ResourceLoader loader = new FileSystemResourceLoader();
			return loader.getResource(resourceSpec);
		}
	}

	private String getResourceName(Resource resource) {
		try {
			return resource.getFilename();
		} catch (IllegalStateException exc) {
			return "unnamed";
		}
	}

	private Pair<String, String> parseResourceSpec(String resourceSpec) {
		int p = resourceSpec.indexOf(':');
		if (p == -1 || p == 0 || p >= resourceSpec.length() - 1) {
			throw new IllegalArgumentException("Invalid resource URL " + resourceSpec);
		}
		return new Pair<String, String>(resourceSpec.substring(0, p), resourceSpec.substring(p + 1));
	}

	private int executeCommand(HostSession s, Map<String, String> commandVariables, StepExecutionContext ctx) {
		CommandExecutionCallbackHandler handler = new StepExecutionContextCallbackHandler(ctx);
		String[] c = replaceVariablesInCommand(command, commandVariables);
		return s.execute(handler, inputResponse, c);
	}

	private String[] replaceVariablesInCommand(String[] commandIn, Map<String, String> commandVariables) {
		TemplateResolver r = new TemplateResolver(commandVariables);
		String[] commandOut = new String[commandIn.length];
		for (int i = 0; i < commandIn.length; i++) {
			commandOut[i] = r.resolveStrict(commandIn[i]);
		}
		return commandOut;
	}

	private static Logger logger = Logger.getLogger(HostCommandExecutionStep.class);

}
