/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.util;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;

import com.xebialabs.deployit.ResolutionException;
import com.xebialabs.deployit.mapper.Pair;

public class TemplateResolver {

	public static final String PROPERTY_SEPERATOR = ".";
	public static final String TEMPLATE_POSTFIX = "}";
	public static final String TEMPLATE_PREFIX = "${";

	private Map<String, String> variables;

	public TemplateResolver(Collection<Object> context) {
		this(contextToVariables(context));
	}

	public TemplateResolver(Map<String, String> variables) {
		this.variables = new HashMap<String, String>();
		for (Map.Entry<String, String> each : variables.entrySet()) {
			if (each.getKey() != null) {
				this.variables.put(each.getKey().toLowerCase(), each.getValue());
			}
		}
	}

	public String resolveLenient(String template) {
		return resolveInternal(template, false);
	}

	public String resolveStrict(String template) {
		return resolveInternal(template, true);
	}

	private String resolveInternal(String template, boolean strict) {
		if (template == null) {
			return null;
		}

		StringBuilder value = new StringBuilder();
		String t = template;
		while (t.length() > 0) {
			int startIndex = t.indexOf(TEMPLATE_PREFIX);
			if (startIndex != -1) {
				int endIndex = t.indexOf(TEMPLATE_POSTFIX, startIndex);
				if (endIndex != -1) {
					String propertyToLookFor = t.substring(startIndex + TEMPLATE_PREFIX.length(), endIndex);
					String propertyValue = variables.get(propertyToLookFor.toLowerCase());
					if (StringUtils.isBlank(propertyValue)) {
						if (strict) {
							throw new ResolutionException("Cannot resolve template \"" + template + "\"  because template variable " + propertyToLookFor
									+ " has no value");
						}

						String beforeAndIncludingPlaceholder = t.substring(0, endIndex + 1);
						String afterPlaceholder = t.substring(endIndex + 1);
						value.append(beforeAndIncludingPlaceholder);
						t = afterPlaceholder;
						continue;
					} else {
						String beforePlaceholder = t.substring(0, startIndex);
						String afterPlaceholder = t.substring(endIndex + 1);
						value.append(beforePlaceholder);
						value.append(propertyValue);
						t = afterPlaceholder;
						continue;
					}
				}
			}
			value.append(t);
			t = "";
		}

		return value.toString();
	}

	public static Map<String, String> contextToVariables(Collection<? extends Object> context) {
		Map<String, String> properties = new HashMap<String, String>();
		for (Object each : context) {
			if (each != null) {
				List<String> readableFields = getExposedProperties(each);
				for (String fieldName : readableFields) {
					Pair<List<String>, Object> res = getValueWithSimpleName(each, fieldName);
					if (res != null) {
						Object fieldValue = res.getSecond();
						if (fieldValue != null) {
							List<String> classnames = res.getFirst();
							for (String classname : classnames) {
								properties.put(classname.toLowerCase() + PROPERTY_SEPERATOR + fieldName.toLowerCase(), fieldValue.toString());
							}
						}
					}
				}
			}
		}
		return properties;
	}

	private static BeanInfo getBeanInfo(Object detailsObject) {
		BeanInfo beanInfo = null;
		try {
			beanInfo = Introspector.getBeanInfo(detailsObject.getClass());
		} catch (IntrospectionException e) {
			throw new RuntimeException("Unable to introspect object " + detailsObject, e);
		}
		return beanInfo;
	}

	public static List<String> getExposedProperties(Object detailsObject) {
		return getExposedProperties(getBeanInfo(detailsObject));
	}

	private static List<String> getExposedProperties(BeanInfo beanInfo) {
		List<String> properties = new ArrayList<String>();
		PropertyDescriptor[] javaBeanDescriptor = beanInfo.getPropertyDescriptors();
		for (int i = 0; i < javaBeanDescriptor.length; i++) {
			PropertyDescriptor beanDescriptor = javaBeanDescriptor[i];
			boolean isWriteAble = beanDescriptor.getWriteMethod() != null;
			String property = beanDescriptor.getName();
			// filter out "internal" properties like the 'id' revision' etc
			if (isWriteAble && !property.equals("id") && !property.equals("class") && !property.equals("groupRevision") && !property.equals("handle")
					&& !property.equals("revision") && !property.equals("typeDescriptor") && !property.equals("type")) {
				properties.add(beanDescriptor.getName().toLowerCase());
			}
		}
		properties.add("label");
		properties.add("description");
		return properties;
	}

	public static Pair<List<String>, Object> getValueWithSimpleName(Object anObject, String fieldName) {
		BeanInfo beanInfo = getBeanInfo(anObject);
		for (PropertyDescriptor descriptor : beanInfo.getPropertyDescriptors()) {
			String name = descriptor.getName();
			if (name.equalsIgnoreCase(fieldName)) {
				Method readMethod = descriptor.getReadMethod();
				try {
					if (readMethod != null) {
						// String declaringClassSimpleName =
						// readMethod.getDeclaringClass().getSimpleName();
						Class<?> currentClazz = anObject.getClass();
						Class<?> clazz = readMethod.getDeclaringClass();
						Object value = readMethod.invoke(anObject, (Object[]) null);
						// now for each class in the hierarchy currentClazz uto
						// and including clazz
						List<String> classNames = makeClassNames(currentClazz, clazz);
						return new Pair<List<String>, Object>(classNames, value);
					}
				} catch (Throwable exc) {
					throw new RuntimeException("Could not get value for " + anObject.getClass() + "." + fieldName, exc);
				}
			}
		}
		return null;
	}

	private static List<String> makeClassNames(Class<?> currentClazz, Class<?> clazz) {
		List<String> classNames = new ArrayList<String>();
		while (currentClazz != clazz) {
			classNames.add(currentClazz.getSimpleName());
			currentClazz = currentClazz.getSuperclass();
		}
		classNames.add(currentClazz.getSimpleName());
		return classNames;
	}

}
