/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.steps;

import java.util.Map;

import org.apache.commons.lang.StringUtils;

import com.xebialabs.deployit.ResolutionException;
import com.xebialabs.deployit.StepExecutionContext;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.hostsession.HostFile;
import com.xebialabs.deployit.hostsession.HostFileUtils;
import com.xebialabs.deployit.hostsession.HostSession;
import com.xebialabs.deployit.hostsession.HostSessionFactory;
import com.xebialabs.deployit.hostsession.LenientTemplateResolvingHostFileInputTransformer;

/**
 * Steps that copies files and directories from one host to another.
 */
@SuppressWarnings("serial")
public class CopyStep extends HostSessionStep {

	protected String sourceLocation;

	protected String destinationLocation;

	protected Host destinationHost;

	protected Map<String, String> properties;

	private boolean createToplevelDirectory;

	/**
	 * Creates a new <tt>CopyStep</tt> that will not create a top-level directory and that will do no placeholder replacement.
	 * 
	 * @see CopyStep#CopyStep(Host, String, Host, String, Map)
	 */
	public CopyStep(Host sourceHost, String sourceLocation, Host destinationHost, String destinationLocation) {
		this(sourceHost, sourceLocation, destinationHost, destinationLocation, (Map<String, String>) null);
	}

	/**
	 * Creates a new <tt>CopyStep</tt> that copies a file from a location on a source host to a location on a destination host, replacing placeholders in the
	 * source file using the given properties. Placeholder replacement is strict, meaning that any if any placeholders in the source file cannot be replaced
	 * (i.e. no corresponding value is found) an error is thrown.
	 * <p/>
	 * Assumes that the source file is a <em>text file</em>, not a directory or other type of file that cannot be read into a string.
	 * 
	 * @param sourceHost
	 *            the source host
	 * @param sourceLocation
	 *            the location on the source host
	 * @param destinationHost
	 *            the destination host
	 * @param destinationLocation
	 *            the location on the destination host
	 * @param createToplevelDirectory
	 *            indicates if the top-level directory should be created if non-existent
	 */
	public CopyStep(Host sourceHost, String sourceLocation, Host destinationHost, String destinationLocation, boolean createToplevelDirectory) {
		this(sourceHost, sourceLocation, destinationHost, destinationLocation, (Map<String, String>) null);
		this.createToplevelDirectory = createToplevelDirectory;
	}

	/**
	 * Creates a new <tt>CopyStep</tt> that copies a file from a location on a source host to a location on a destination host, replacing placeholders in the
	 * source file using the given properties. Placeholder replacement is strict, meaning that any if any placeholders in the source file cannot be replaced
	 * (i.e. no corresponding value is found) an error is thrown.
	 * <p/>
	 * Assumes that the source file is a <em>text file</em>, not a directory or other type of file that cannot be read into a string.
	 * 
	 * @param sourceHost
	 *            the source host
	 * @param sourceLocation
	 *            the location on the source host
	 * @param destinationHost
	 *            the destination host
	 * @param destinationLocation
	 *            the location on the destination host
	 * @param properties
	 *            the properties to be used for placeholder replacement
	 */
	/*
	 * XXX: debatable if this should go here or be a separate subclass. It's also worth considering whether the CopyStep should accept the
	 * ConfigurationProperties CI (and thus needs to know how to get from there to a TemplateResolver), or whether the *runbook* should construct the
	 * TemplateResolver and pass it in. Of course, the latter currently isn't an option since the resolver isn't serializable...
	 */
	public CopyStep(Host sourceHost, String sourceLocation, Host destinationHost, String destinationLocation, Map<String, String> properties) {
		super(sourceHost);
		if (StringUtils.isEmpty(sourceLocation)) {
			throw new ResolutionException("No source location specified");
		} else if (StringUtils.isEmpty(destinationLocation)) {
			throw new ResolutionException("No destination location specified");
		}
		this.sourceLocation = sourceLocation;
		this.destinationLocation = destinationLocation;

		this.destinationHost = destinationHost;

		this.properties = properties;

		String description = "Copy " + sourceLocation + " from " + sourceHost + " to " + destinationLocation + " on " + destinationHost;
		if (this.properties != null) {
			description += " while replacing property values";
		}
		setDescription(description);
	}

	public boolean execute(StepExecutionContext ctx) {
		HostSession sourceSession = getHostSession();
		HostSession destinationSession = HostSessionFactory.getHostSession(destinationHost);
		try {

			HostFile sourceFile = sourceSession.getFile(sourceLocation);
			HostFile destinationFile = destinationSession.getFile(destinationLocation);

			if (createToplevelDirectory && !destinationFile.exists()) {
				if (sourceFile.isDirectory()) {
					ctx.logOutput("Creating destination directory " + destinationFile);
					destinationFile.mkdirs();

				} else {
					HostFile parentFile = destinationFile.getParentFile();
					ctx.logOutput("Creating destination directory " + parentFile);
					parentFile.mkdirs();
				}
			}

			if (properties != null) {
				HostFileUtils.copy(sourceFile, destinationFile, new LenientTemplateResolvingHostFileInputTransformer(properties));
			} else {
				HostFileUtils.copy(sourceFile, destinationFile);
			}

			return true;
		} finally {
			sourceSession.close();
			destinationSession.close();
		}
	}

	@Override
	public boolean equals(Object o) {
		if (this == o)
			return true;
		if (!(o instanceof CopyStep))
			return false;

		CopyStep copyStep = (CopyStep) o;

		if (!destinationHost.equals(copyStep.destinationHost))
			return false;
		if (!destinationLocation.equals(copyStep.destinationLocation))
			return false;
		if (properties != null ? !properties.equals(copyStep.properties) : copyStep.properties != null)
			return false;
		if (!sourceLocation.equals(copyStep.sourceLocation))
			return false;

		return true;
	}

	@Override
	public int hashCode() {
		int result = sourceLocation.hashCode();
		result = 31 * result + destinationLocation.hashCode();
		result = 31 * result + destinationHost.hashCode();
		result = 31 * result + (properties != null ? properties.hashCode() : 0);
		return result;
	}

}
