/*
 * Copyright (c) 2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to  (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.ci;

import static com.google.common.base.Preconditions.checkNotNull;

import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Predicate;

import com.xebialabs.deployit.BaseConfigurationItem;
import com.xebialabs.deployit.ConfigurationItem;
import com.xebialabs.deployit.ConfigurationItemProperty;
import com.xebialabs.deployit.ci.artifact.DeployableArtifact;

/**
 * Represents a collection of artifacts and resources that can be deployed onto middleware.
 */
@SuppressWarnings("serial")
@ConfigurationItem(expandable = true, description = "A package of a certion version of an application, i.e. a grouping of a number of deployable artifact CI's. Contains as its members all the application components that make it up. Is contained by the Application CI of which it supplies a version.")
public class DeploymentPackage extends BaseConfigurationItem {

	@ConfigurationItemProperty(required = false, description = "The version of the application package. Examples; 1.0, 1.2.3, 1.0-SNAPSHOT, 3.4-MILESTONE-RC-3")
	private String version;

	@ConfigurationItemProperty(required = true, description = "The application the package is a part of.")
	private Application application;

	@ConfigurationItemProperty(required = false, description = "The set of deployable artifacts that are part of this package")
	private Set<DeployableArtifact> deployableArtifacts = new HashSet<DeployableArtifact>();

	@ConfigurationItemProperty(description = "The set of middleware resources that are part of this package")
	private Set<Serializable> middlewareResources = new HashSet<Serializable>();

	public DeploymentPackage() {
		// no op
	}

	public DeploymentPackage(DeployableArtifact... artifacts) {
		StringBuilder labelBuilder = new StringBuilder();
		for (DeployableArtifact each : artifacts) {
			addDeployableArtifact(each);
			if (labelBuilder.length() > 0) {
				labelBuilder.append(" and ");
			}
			labelBuilder.append(each);
		}
		setLabel("DeploymentPackage containing " + labelBuilder);
	}

	public Collection<Serializable> getAllMembers() {
		return getMembersOfType(Serializable.class);
	}

	@SuppressWarnings("unchecked")
	public <T extends Serializable> Collection<T> getMembersOfType(final Class<T> ciType) {
		Predicate typeSelector = new Predicate() {
			public boolean evaluate(Object ci) {
				return ciType.isAssignableFrom(ci.getClass());
			}

		};
		Collection<T> members = CollectionUtils.select(deployableArtifacts, typeSelector);
		members.addAll(CollectionUtils.select(middlewareResources, typeSelector));
		return members;
	}

	public String getVersion() {
		return version;
	}

	public void setVersion(String version) {
		this.version = version;
	}

	public Application getApplication() {
		return application;
	}

	public void setApplication(Application application) {
		this.application = application;
	}

	public Set<DeployableArtifact> getDeployableArtifacts() {
		return Collections.unmodifiableSet(deployableArtifacts);
	}

	public void setDeployableArtifacts(Set<DeployableArtifact> deployableArtifacts) {
		this.deployableArtifacts = deployableArtifacts;
	}

	public void addDeployableArtifact(DeployableArtifact deployableArtifact) {
		deployableArtifacts.add(checkNotNull(deployableArtifact));
	}

	public Set<Serializable> getMiddlewareResources() {
		return Collections.unmodifiableSet(middlewareResources);
	}

	public void setMiddlewareResources(Set<Serializable> middlewareResources) {
		this.middlewareResources = middlewareResources;
	}

	public void addMiddlewareResource(Serializable middlewareResource) {
		middlewareResources.add(checkNotNull(middlewareResource));
	}

}
