/*
 * Copyright (c) 2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to  (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit;

import static com.google.common.base.Preconditions.checkNotNull;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Holds information on what steps to be executed (in the middleware) to make
 * the changes "happen".
 */
public class ChangeResolution {

	private Set<Change<?>> changes = new HashSet<Change<?>>();

	private List<Step> steps = new ArrayList<Step>();

	/**
	 * Creates an empty change resolution.
	 */
	public ChangeResolution() {
	}

	/**
	 * Creates a change resolution for the resolved change. Additional changes
	 * can also be added by invoking {@link #addChange(Change)}.
	 * 
	 * @param c
	 *            the resolved change (may not be <tt>null</tt>)
	 */
	public ChangeResolution(Change<?> c) {
		changes.add(checkNotNull(c, "Change may not be null"));
	}

	/**
	 * Creates a change resolution linking the given change with the provided
	 * step list.
	 * 
	 * @param c
	 *            the resolved change (may not be <tt>null</tt>)
	 * @param steps
	 *            the steps for the change (may not be <tt>null</tt>)
	 */
	public ChangeResolution(Change<?> c, List<Step> steps) {
		changes.add(checkNotNull(c, "Change may not be null"));
		this.steps.addAll(checkNotNull(steps, "List of Steps may not be null"));
	}

	/**
	 * Returns the changes that are resolved by this changeresolution.
	 * 
	 * @return the set of resolved changes.
	 */
	public Set<Change<?>> getChanges() {
		return Collections.unmodifiableSet(changes);
	}

	/**
	 * Returns the steps that need to executed for the resolved changes.
	 * 
	 * @return the list of steps (never <tt>null</tt>).
	 */
	public List<Step> getSteps() {
		return Collections.unmodifiableList(steps);
	}

	/**
	 * Adds a change to the list of changes resolved.
	 * 
	 * @param changeToAdd
	 *            the change to add (may not be <tt>null</tt>).
	 */
	public void addChange(Change<?> changeToAdd) {
		changes.add(checkNotNull(changeToAdd, "Change may not be null"));
	}

	/**
	 * Adds a step to the list of steps to be executed.
	 * 
	 * @param stepToAdd
	 *            the step to add (may not be <tt>null</tt>).
	 */
	public void addStep(Step stepToAdd) {
		steps.add(checkNotNull(stepToAdd, "Step may not be null"));
	}

	/**
	 * Adds multiple steps to the list of steps to be executed.
	 * 
	 * @param stepsToAdd
	 *            the steps to add (may not be <tt>null</tt>).
	 */
	public void addSteps(List<Step> stepsToAdd) {
		steps.addAll(checkNotNull(stepsToAdd, "List of Steps may not be null"));
	}

}
