package com.xebialabs.deployit.inspection;

import static com.google.common.collect.Lists.newArrayList;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


import com.xebialabs.deployit.plugin.api.execution.ExecutionContext;
import com.xebialabs.deployit.plugin.api.execution.Step;
import com.xebialabs.deployit.plugin.api.execution.Step.Result;
import com.xebialabs.deployit.plugin.api.inspection.Inspect;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

/**
 * Discovers ConfigurationItems from reality and/or syncs them.
 */
public class Inspector {

    public List<ConfigurationItem> inspect(ConfigurationItem item, ExecutionContext originalContext) {
		InspectionContext ctx = new InspectionContext(originalContext);
		return doInspect(item, ctx);
	}

    public List<ConfigurationItem> inspect(ConfigurationItem item, Map<String, Object> taskAttributes) {
		InspectionContext ctx = new InspectionContext(taskAttributes);
		return doInspect(item, ctx);
	}

	@SuppressWarnings({ "unchecked", "rawtypes" })
	private List<ConfigurationItem> doInspect(ConfigurationItem item, InspectionContext ctx) {
		inspectConfigurationItem(item, ctx);
		Step step;
		while ((step = ctx.getNextStep()) != null) {
			logger.debug("Invoking inspection step: {}", step.getDescription());

			Result result;
			try {
				ctx.startStepExecution(step);
				result = step.execute(ctx);
			} catch (Exception exc) {
				String lastErrorMessage = getLastMessageFromLogs(ctx.getCapturedError());
				throw new RuntimeException(String.format("Step '%s' failed with error: %s", step.getDescription(), lastErrorMessage));
			}

			if (result == Result.Fail) {
				String lastErrorMessage = getLastMessageFromLogs(ctx.getCapturedError());
				throw new RuntimeException(String.format("Step '%s' failed with error: %s", step.getDescription(), lastErrorMessage));
			}

			inspectNewDiscoveredConfigurationItems(ctx);
		}

		return ctx.getInspected();
	}

	private String getLastMessageFromLogs(List<String> messages) {
		if(messages.size() > 0) {
			ListIterator<String> reverseIterator = messages.listIterator(messages.size());
			while(reverseIterator.hasPrevious()) {
				  String message = reverseIterator.previous();
				  if(message != null && message.length() != 0) {
					  return message;
				  }
				}
		}
	    
	    return "No log found!";
    }
	
	private void inspectNewDiscoveredConfigurationItems(InspectionContext context) {
		for (ConfigurationItem configurationItem : newArrayList(context.getDiscovered())) {
			inspectConfigurationItem(configurationItem, context);
		}
	}

	private void inspectConfigurationItem(ConfigurationItem item, InspectionContext context) {		
		checkRequiredDiscoveryProperties(item);
		Method inspectMethod = findInspectMethod(item);
		if(inspectMethod != null) {
			try {
				logger.debug("Invoking @Inspect method on {}", item);
	            inspectMethod.invoke(item, context);
            } catch (IllegalArgumentException e) {
            	throw new RuntimeException("Cannot invoke @Inspect method on " + item, e);
            } catch (IllegalAccessException e) {
            	throw new RuntimeException("Cannot invoke @Inspect method on " + item, e);
            } catch (InvocationTargetException e) {
            	throw new RuntimeException("Error invoking @Inspect method on " + item, e);
            }
		}
		context.inspected(item);
	}

	private void checkRequiredDiscoveryProperties(ConfigurationItem item) {
		Descriptor descriptor = DescriptorRegistry.getDescriptor(item.getType());
		for (PropertyDescriptor propertyDescriptor : descriptor.getPropertyDescriptors()) {
			if (propertyDescriptor.isInspectionProperty() && propertyDescriptor.isRequiredForInspection()) {
				Object o = propertyDescriptor.get(item);
				if (o == null) {
					throw new IllegalArgumentException("Missing required property for discovery " + propertyDescriptor.getName());
				}
			}
		}
	}

	private Method findInspectMethod(ConfigurationItem item) {
		Method[] methods = item.getClass().getMethods();		
		for (Method method : methods) {		
			if (method.isAnnotationPresent(Inspect.class)) {
				return method;
			}
		}
		return null;
	}
	
	private Logger logger = LoggerFactory.getLogger(Inspector.class);

}
